package cn.boboweike.carrot.storage;

import cn.boboweike.carrot.scheduling.partition.Partitioner;
import cn.boboweike.carrot.storage.listeners.StorageProviderChangeListener;
import cn.boboweike.carrot.tasks.RecurringTask;
import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.tasks.mappers.TaskMapper;
import cn.boboweike.carrot.tasks.states.StateName;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskDetails;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

public interface PartitionedStorageProvider {

    int BATCH_SIZE = 5000;
    int DEFAULT_NUM_OF_PARTITIONS = 1;

    String PARTITION_PREFIX = "partition_";

    String PARTITION_HINT_KEY = "partition_hint_key";

    String getName();

    int getTotalNumOfPartitions();

    boolean lockByPartition(Integer partition, int durationInSeconds, String lockedBy);

    boolean extendLockByPartition(Integer partition, int durationInSeconds, String lockedBy);

    boolean unlockByPartition(Integer partition);

    void setTaskMapper(TaskMapper taskMapper);

    void setPartitioner(Partitioner partitioner);

    /**
     * This method allows to reinitialize the StorageProvider.
     * It can be used if you are using Flyway or Liquibase to setup your database manually.
     *
     * By default, this method is automatically called on construction of the StorageProvider
     *
     * @param databaseOptions defines whether to set up the StorageProvider or validate whether the StorageProvider is set up correctly.
     */
    void setUpStorageProvider(StorageProviderUtils.DatabaseOptions databaseOptions);

    void addTaskStorageOnChangeListener(StorageProviderChangeListener listener);

    void removeTaskStorageOnChangeListener(StorageProviderChangeListener listener);

    void announceBackgroundTaskServer(BackgroundTaskServerStatus serverStatus);

    boolean signalBackgroundTaskServerAlive(BackgroundTaskServerStatus serverStatus);

    void signalBackgroundTaskServerStopped(BackgroundTaskServerStatus serverStatus);

    List<BackgroundTaskServerStatus> getBackgroundTaskServers();

    UUID getLongestRunningBackgroundTaskServerId();

    int removeTimedOutBackgroundTaskServers(Instant heartbeatOlderThan);

    void saveMetadata(CarrotMetadata metadata);

    List<CarrotMetadata> getMetadata(String name);

    CarrotMetadata getMetadata(String name, String owner);

    void deleteMetadata(String name);

    Task save(Task task);

    Task saveByPartition(Task task, Integer partition);

    int deletePermanentlyByPartition(UUID id, Integer partition);

    Task getTaskById(UUID id);

    List<Task> save(List<Task> tasks);

    List<Task> saveByPartition(List<Task> tasks, Integer partition);

    List<Task> getTasksByPartition(StateName state, Instant updatedBefore, PageRequest pageRequest, Integer partition);

    List<Task> getScheduledTasksByPartition(Instant scheduledBefore, PageRequest pageRequest, Integer partition);

    List<Task> getTasksByPartition(StateName state, PageRequest pageRequest, Integer partition);

    Page<Task> getTaskPageByPartition(StateName state, PageRequest pageRequest, Integer partition);

    int deleteTasksPermanentlyByPartition(StateName state, Instant updatedBefore, Integer partition);

    Set<String> getDistinctTaskSignatures(StateName... states);

    boolean existsByPartition(TaskDetails taskDetails, Integer partition, StateName... states);

    boolean recurringTaskExistsByPartition(String recurringTaskId, Integer partition, StateName... states);

    RecurringTask saveRecurringTask(RecurringTask recurringTask);

    List<RecurringTask> getRecurringTasksByPartition(Integer partition);

    List<RecurringTask> getRecurringTasks();

    long countRecurringTasksByPartition(Integer partition);

    int deleteRecurringTask(String id);

    TaskStatsData getTaskStatsData();

    void publishTotalAmountOfSucceededTasks(int amount);

    default Task getTaskById(TaskId taskId) {
        return getTaskById(taskId.asUUID());
    }

    void close();
}
