package cn.boboweike.carrot.storage;

import cn.boboweike.carrot.tasks.Task;

import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import static cn.boboweike.carrot.utils.StringUtils.isNullOrEmpty;
import static java.util.stream.Collectors.toList;

public class StorageProviderUtils {

    private StorageProviderUtils() {
    }

    public static String elementPrefixer(String prefix, String element) {
        return isNullOrEmpty(prefix) ? element : prefix + element;
    }

    public static String elementPrefixerWithPartition(String prefix, String element, Integer partition) {
        String result = elementPrefixer(prefix, element);
        result += "_" + partition;
        return result;
    }

    public enum DatabaseOptions {
        CREATE,
        SKIP_CREATE
    }

    private static final String FIELD_ID = "id";

    public static final class Migrations {
        private Migrations() {
        }

        public static final String NAME = "migrations";
        public static final String FIELD_ID = StorageProviderUtils.FIELD_ID;
        public static final String FIELD_NAME = "name";
        public static final String FIELD_DATE = "date";
        public static final String FIELD_NUM_OF_PARTITIONS = "num_of_partitions";
    }

    public static final class Tasks {
        private Tasks() {
        }

        public static final String NAME = "tasks";
        public static final String FIELD_ID = StorageProviderUtils.FIELD_ID;
        public static final String FIELD_VERSION = "version";
        public static final String FIELD_STATE = "state";
        public static final String FIELD_TASK_AS_JSON = "taskAsJson";
        public static final String FIELD_TASK_SIGNATURE = "taskSignature";
        public static final String FIELD_CREATED_AT = "createdAt";
        public static final String FIELD_UPDATED_AT = "updatedAt";
        public static final String FIELD_SCHEDULED_AT = "scheduledAt";
        public static final String FIELD_RECURRING_TASK_ID = "recurringTaskId";
    }

    public static class RecurringTasks {
        private RecurringTasks() {
        }

        public static final String NAME = "recurring_tasks";
        public static final String FIELD_ID = StorageProviderUtils.FIELD_ID;
        public static final String FIELD_VERSION = "version";
        public static final String FIELD_TASK_AS_JSON = "taskAsJson";
    }

    public static class ShedLock {
        private ShedLock() {

        }
        public static final String NAME = "shed_lock";
    }

    public static final class BackgroundTaskServers {
        private BackgroundTaskServers() {
        }

        public static final String NAME = "background_task_servers";
        public static final String FIELD_ID = StorageProviderUtils.FIELD_ID;
        public static final String FIELD_WORKER_POOL_SIZE = "workerPoolSize";
        public static final String FIELD_POLL_INTERVAL_IN_SECONDS = "pollIntervalInSeconds";
        public static final String FIELD_DELETE_SUCCEEDED_TASKS_AFTER = "deleteSucceededTasksAfter";
        public static final String FIELD_PERMANENTLY_DELETE_TASKS_AFTER = "permanentlyDeleteTasksAfter";
        public static final String FIELD_DELETE_DELETED_TASKS_AFTER = "permanentlyDeleteDeletedTasksAfter";
        public static final String FIELD_FIRST_HEARTBEAT = "firstHeartbeat";
        public static final String FIELD_LAST_HEARTBEAT = "lastHeartbeat";
        public static final String FIELD_IS_RUNNING = "running";
        public static final String FIELD_SYSTEM_TOTAL_MEMORY = "systemTotalMemory";
        public static final String FIELD_SYSTEM_FREE_MEMORY = "systemFreeMemory";
        public static final String FIELD_SYSTEM_CPU_LOAD = "systemCpuLoad";
        public static final String FIELD_PROCESS_MAX_MEMORY = "processMaxMemory";
        public static final String FIELD_PROCESS_FREE_MEMORY = "processFreeMemory";
        public static final String FIELD_PROCESS_ALLOCATED_MEMORY = "processAllocatedMemory";
        public static final String FIELD_PROCESS_CPU_LOAD = "processCpuLoad";
        public static final String FIELD_PARTITION = "partition";
    }

    public static final class Metadata {
        private Metadata() {
        }

        public static final String METADATA_OWNER_CLUSTER = "cluster";

        public static final String NAME = "metadata";
        public static final String FIELD_ID = StorageProviderUtils.FIELD_ID;
        public static final String STATS_ID = "succeeded-tasks-counter-cluster";
        public static final String STATS_NAME = "succeeded-tasks-counter";
        public static final String STATS_OWNER = "cluster";
        public static final String FIELD_NAME = "name";
        public static final String FIELD_OWNER = "owner";
        public static final String FIELD_VALUE = "value";
        public static final String FIELD_CREATED_AT = "createdAt";
        public static final String FIELD_UPDATED_AT = "updatedAt";

    }

    public static List<Task> returnConcurrentModifiedTasks(List<Task> tasks, Consumer<Task> consumer) {
        return tasks.stream()
                .map(toConcurrentTaskModificationExceptionIfFailed(consumer))
                .filter(Objects::nonNull)
                .map(ConcurrentTaskModificationException.class::cast)
                .flatMap(ex -> ex.getConcurrentUpdatedTasks().stream())
                .collect(toList());
    }

    public static Function<Task, Exception> toConcurrentTaskModificationExceptionIfFailed(Consumer<Task> test) {
        return task -> {
            try {
                test.accept(task);
                return null;
            } catch (ConcurrentTaskModificationException e) {
                return e;
            }
        };
    }
}

