package cn.boboweike.carrot.storage;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.locks.ReentrantLock;

import static java.lang.Math.ceil;
import static java.math.BigDecimal.ZERO;
import static java.math.BigDecimal.valueOf;

/**
 * Class which takes TaskStats and extends them with estimations on how long the work will take based on previous TaskStats.
 */
public class TaskStatsEnricher {
    private final ReentrantLock lock = new ReentrantLock();
    private TaskStats firstRelevantTaskStats;
    private TaskStats previousTaskStats;
    private TaskStatsExtended taskStatsExtended;

    public TaskStatsExtended enrich(TaskStats taskStats) {
        TaskStats latestTaskStats = getLatestTaskStats(taskStats, previousTaskStats);
        if (lock.tryLock()) {
            setFirstRelevantTaskStats(latestTaskStats);
            setTaskStatsExtended(latestTaskStats);
            setPreviousTaskStats(latestTaskStats);
            lock.unlock();
        }
        return taskStatsExtended;
    }

    private static TaskStats getLatestTaskStats(TaskStats taskStats, TaskStats previousTaskStats) {
        if (previousTaskStats == null) return taskStats;
        if (taskStats.getTimeStamp().isAfter(previousTaskStats.getTimeStamp())) return taskStats;
        return previousTaskStats;
    }

    private void setFirstRelevantTaskStats(TaskStats taskStats) {
        if (firstRelevantTaskStats == null
                || (taskStats.getEnqueued() < 1 && taskStats.getProcessing() < 1)
                || (taskStats.getEnqueued() > firstRelevantTaskStats.getEnqueued())) {
            firstRelevantTaskStats = taskStats;
        }
    }

    private void setTaskStatsExtended(TaskStats taskStats) {
        TaskStats actualPreviousTaskStats = this.previousTaskStats != null ? this.previousTaskStats : this.firstRelevantTaskStats;
        Long amountSucceeded = taskStats.getSucceeded() - actualPreviousTaskStats.getSucceeded();
        Long amountFailed = taskStats.getFailed() - actualPreviousTaskStats.getFailed();
        Instant estimatedProcessingFinishedInstant = estimatedProcessingFinishedInstant(firstRelevantTaskStats, taskStats);
        if (estimatedProcessingFinishedInstant != null) {
            taskStatsExtended = new TaskStatsExtended(taskStats, amountSucceeded, amountFailed, estimatedProcessingFinishedInstant);
        } else if (taskStatsExtended != null && taskStatsExtended.getEstimation().isEstimatedProcessingFinishedInstantAvailable()) {
            taskStatsExtended = new TaskStatsExtended(taskStats, amountSucceeded, amountFailed, taskStatsExtended.getEstimation().getEstimatedProcessingFinishedAt());
        } else {
            taskStatsExtended = new TaskStatsExtended(taskStats);
        }
    }

    private Instant estimatedProcessingFinishedInstant(TaskStats firstRelevantTaskStats, TaskStats taskStats) {
        if (taskStats.getSucceeded() - firstRelevantTaskStats.getSucceeded() < 1) return null;
        BigDecimal durationForAmountSucceededInSeconds = valueOf(Duration.between(firstRelevantTaskStats.getTimeStamp(), taskStats.getTimeStamp()).getSeconds());
        if (ZERO.equals(durationForAmountSucceededInSeconds)) return null;
        BigDecimal amountSucceededPerSecond = valueOf(ceil(taskStats.getSucceeded() - firstRelevantTaskStats.getSucceeded())).divide(durationForAmountSucceededInSeconds, RoundingMode.CEILING);
        if (ZERO.equals(amountSucceededPerSecond)) return null;
        BigDecimal processingTimeInSeconds = BigDecimal.valueOf(taskStats.getEnqueued() + taskStats.getProcessing()).divide(amountSucceededPerSecond, RoundingMode.HALF_UP);
        return Instant.now().plusSeconds(processingTimeInSeconds.longValue());
    }

    private void setPreviousTaskStats(TaskStats taskStats) {
        previousTaskStats = taskStats;
    }
}
