package cn.boboweike.carrot.storage;

import cn.boboweike.carrot.scheduling.partition.Partitioner;
import cn.boboweike.carrot.storage.listeners.StorageProviderChangeListener;
import cn.boboweike.carrot.tasks.RecurringTask;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.mappers.TaskMapper;
import cn.boboweike.carrot.tasks.states.StateName;
import cn.boboweike.carrot.utils.resilience.Lock;
import cn.boboweike.carrot.utils.resilience.MultiLock;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

public class ThreadSafePartitionedStorageProvider implements PartitionedStorageProvider {

    private final PartitionedStorageProvider storageProvider;

    public ThreadSafePartitionedStorageProvider(PartitionedStorageProvider storageProvider) {
        this.storageProvider = storageProvider;
    }


    @Override
    public String getName() {
        return storageProvider.getName();
    }

    @Override
    public int getTotalNumOfPartitions() {
        return storageProvider.getTotalNumOfPartitions();
    }

    @Override
    public boolean lockByPartition(Integer partition, int durationInSeconds, String lockedBy) {
        return storageProvider.lockByPartition(partition, durationInSeconds, lockedBy);
    }

    @Override
    public boolean extendLockByPartition(Integer partition, int durationInSeconds, String lockedBy) {
        return storageProvider.extendLockByPartition(partition, durationInSeconds, lockedBy);
    }

    @Override
    public boolean unlockByPartition(Integer partition) {
        return storageProvider.unlockByPartition(partition);
    }

    @Override
    public void setTaskMapper(TaskMapper taskMapper) {
        storageProvider.setTaskMapper(taskMapper);
    }

    @Override
    public void setPartitioner(Partitioner partitioner) {
        storageProvider.setPartitioner(partitioner);
    }

    @Override
    public void setUpStorageProvider(StorageProviderUtils.DatabaseOptions databaseOptions) {
        storageProvider.setUpStorageProvider(databaseOptions);
    }

    @Override
    public void addTaskStorageOnChangeListener(StorageProviderChangeListener listener) {
        storageProvider.addTaskStorageOnChangeListener(listener);
    }

    @Override
    public void removeTaskStorageOnChangeListener(StorageProviderChangeListener listener) {
        storageProvider.removeTaskStorageOnChangeListener(listener);
    }

    @Override
    public void announceBackgroundTaskServer(BackgroundTaskServerStatus serverStatus) {
        storageProvider.announceBackgroundTaskServer(serverStatus);
    }

    @Override
    public boolean signalBackgroundTaskServerAlive(BackgroundTaskServerStatus serverStatus) {
        return storageProvider.signalBackgroundTaskServerAlive(serverStatus);
    }

    @Override
    public void signalBackgroundTaskServerStopped(BackgroundTaskServerStatus serverStatus) {
        storageProvider.signalBackgroundTaskServerStopped(serverStatus);
    }

    @Override
    public List<BackgroundTaskServerStatus> getBackgroundTaskServers() {
        return storageProvider.getBackgroundTaskServers();
    }

    @Override
    public UUID getLongestRunningBackgroundTaskServerId() {
        return storageProvider.getLongestRunningBackgroundTaskServerId();
    }

    @Override
    public int removeTimedOutBackgroundTaskServers(Instant heartbeatOlderThan) {
        return storageProvider.removeTimedOutBackgroundTaskServers(heartbeatOlderThan);
    }

    @Override
    public void saveMetadata(CarrotMetadata metadata) {
        storageProvider.saveMetadata(metadata);
    }

    @Override
    public List<CarrotMetadata> getMetadata(String key) {
        return storageProvider.getMetadata(key);
    }

    @Override
    public CarrotMetadata getMetadata(String name, String owner) {
        return storageProvider.getMetadata(name, owner);
    }

    @Override
    public void deleteMetadata(String name) {
        storageProvider.deleteMetadata(name);
    }

    @Override
    public Task save(Task task) {
        try (Lock lock = task.lock()) {
            return storageProvider.save(task);
        }
    }

    @Override
    public Task saveByPartition(Task task, Integer partition) {
        try (Lock lock = task.lock()) {
            return storageProvider.saveByPartition(task, partition);
        }
    }

    @Override
    public int deletePermanentlyByPartition(UUID id, Integer partition) {
        return storageProvider.deletePermanentlyByPartition(id, partition);
    }

    @Override
    public Task getTaskById(UUID id) {
        return storageProvider.getTaskById(id);
    }

    @Override
    public List<Task> save(List<Task> tasks) {
        try (MultiLock lock = new MultiLock(tasks)) {
            return storageProvider.save(tasks);
        }
    }

    @Override
    public List<Task> saveByPartition(List<Task> tasks, Integer partition) {
        try (MultiLock lock = new MultiLock(tasks)) {
            return storageProvider.saveByPartition(tasks, partition);
        }
    }

    @Override
    public List<Task> getTasksByPartition(StateName state, Instant updatedBefore, PageRequest pageRequest, Integer partition) {
        return storageProvider.getTasksByPartition(state, updatedBefore, pageRequest, partition);
    }

    @Override
    public List<Task> getScheduledTasksByPartition(Instant scheduledBefore, PageRequest pageRequest, Integer partition) {
        return storageProvider.getScheduledTasksByPartition(scheduledBefore, pageRequest, partition);
    }

    @Override
    public List<Task> getTasksByPartition(StateName state, PageRequest pageRequest, Integer partition) {
        return storageProvider.getTasksByPartition(state, pageRequest, partition);
    }

    @Override
    public Page<Task> getTaskPageByPartition(StateName state, PageRequest pageRequest, Integer partition) {
        return storageProvider.getTaskPageByPartition(state, pageRequest, partition);
    }

    @Override
    public int deleteTasksPermanentlyByPartition(StateName state, Instant updatedBefore, Integer partition) {
        return storageProvider.deleteTasksPermanentlyByPartition(state, updatedBefore, partition);
    }

    @Override
    public Set<String> getDistinctTaskSignatures(StateName... states) {
        return storageProvider.getDistinctTaskSignatures(states);
    }

    @Override
    public boolean existsByPartition(TaskDetails taskDetails, Integer partition, StateName... states) {
        return storageProvider.existsByPartition(taskDetails, partition, states);
    }

    @Override
    public boolean recurringTaskExistsByPartition(String recurringTaskId, Integer partition, StateName... states) {
        return storageProvider.recurringTaskExistsByPartition(recurringTaskId, partition, states);
    }

    @Override
    public RecurringTask saveRecurringTask(RecurringTask recurringTask) {
        return storageProvider.saveRecurringTask(recurringTask);
    }

    @Override
    public List<RecurringTask> getRecurringTasksByPartition(Integer partition) {
        return storageProvider.getRecurringTasksByPartition(partition);
    }

    @Override
    public List<RecurringTask> getRecurringTasks() {
        return storageProvider.getRecurringTasks();
    }

    @Override
    public long countRecurringTasksByPartition(Integer partition) {
        return storageProvider.countRecurringTasksByPartition(partition);
    }

    @Override
    public int deleteRecurringTask(String id) {
        return storageProvider.deleteRecurringTask(id);
    }

    @Override
    public TaskStatsData getTaskStatsData() {
        return storageProvider.getTaskStatsData();
    }

    @Override
    public void publishTotalAmountOfSucceededTasks(int amount) {
        storageProvider.publishTotalAmountOfSucceededTasks(amount);
    }

    @Override
    public void close() {
        storageProvider.close();
    }

    public PartitionedStorageProvider getStorageProvider() {
        return storageProvider;
    }
}
