package cn.boboweike.carrot.storage.metrics;

import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.storage.TaskStats;
import cn.boboweike.carrot.storage.TaskStatsData;
import cn.boboweike.carrot.storage.listeners.TaskStatsChangeListener;
import cn.boboweike.carrot.tasks.states.StateName;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tags;

import java.util.concurrent.atomic.AtomicLong;

public class StorageProviderMetricsBinder implements TaskStatsChangeListener, AutoCloseable {

    private final PartitionedStorageProvider storageProvider;
    private final MeterRegistry meterRegistry;
    private final AtomicLong scheduledGauge = new AtomicLong(0);
    private final AtomicLong enqueuedGauge = new AtomicLong(0);
    private final AtomicLong processingGauge = new AtomicLong(0);
    private final AtomicLong failedGauge = new AtomicLong(0);
    private final AtomicLong succeededGauge = new AtomicLong(0);
    private final AtomicLong allTimeSucceededGauge = new AtomicLong(0);
    private final AtomicLong deletedGauge = new AtomicLong(0);


    public StorageProviderMetricsBinder(PartitionedStorageProvider storageProvider,MeterRegistry meterRegistry) {
        this.storageProvider = storageProvider;
        this.meterRegistry = meterRegistry;
        registerStorageProviderMetrics();
    }

    public void registerStorageProviderMetrics() {
        registerGauge(StateName.SCHEDULED, scheduledGauge);
        registerGauge(StateName.ENQUEUED, enqueuedGauge);
        registerGauge(StateName.PROCESSING, processingGauge);
        registerGauge(StateName.FAILED, failedGauge);
        registerGauge(StateName.SUCCEEDED, succeededGauge);
        registerGauge("ALL_TIME_SUCCEEDED", allTimeSucceededGauge);
        registerGauge(StateName.DELETED, deletedGauge);

        onChange(this.storageProvider.getTaskStatsData());
        this.storageProvider.addTaskStorageOnChangeListener(this);
    }

    private void registerGauge(StateName stateName, AtomicLong number) {
        registerGauge(stateName.toString(), number);
    }

    private void registerGauge(String stateName, AtomicLong number) {
        meterRegistry.gauge("carrot.tasks." + stateName.toLowerCase(), Tags.of("state", stateName), number);
    }

    @Override
    public void onChange(TaskStatsData taskStatsData) {
        TaskStats taskStats = taskStatsData.getOverallTaskStats();
        scheduledGauge.set(taskStats.getScheduled());
        enqueuedGauge.set(taskStats.getEnqueued());
        processingGauge.set(taskStats.getProcessing());
        failedGauge.set(taskStats.getFailed());
        succeededGauge.set(taskStats.getSucceeded());
        allTimeSucceededGauge.set(taskStats.getAllTimeSucceeded());
        deletedGauge.set(taskStats.getDeleted());
    }

    @Override
    public void close() {
        this.storageProvider.removeTaskStorageOnChangeListener(this);
    }
}