package cn.boboweike.carrot.storage.nosql.common;

import cn.boboweike.carrot.CarrotException;
import cn.boboweike.carrot.storage.nosql.common.migrations.NoSqlMigration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Stream;

public abstract class NoSqlDatabaseCreator<T extends NoSqlMigration> {
    private static final Logger LOGGER = LoggerFactory.getLogger(NoSqlDatabaseCreator.class);

    private final List<T> migrations;

    protected NoSqlDatabaseCreator(List<T> migrations) {
        this.migrations = migrations;
    }

    public void runMigrations() {
        getMigrations()
                .filter(this::isValidMigration)
                .sorted(Comparator.comparing(m -> m.getClassName()))
                .forEach(this::runMigrationIfNecessary);
    }

    protected boolean isValidMigration(T noSqlMigration) {
        return true;
    }

    protected abstract boolean isIncreasePartitions(T noSqlMigration);

    protected abstract boolean isNewMigration(T noSqlMigration);

    protected abstract void runMigration(T noSqlMigration) throws Exception;

    protected abstract boolean markMigrationAsDone(T noSqlMigration);

    protected void runMigrationIfNecessary(T noSqlMigration) {
        if (!isNewMigration(noSqlMigration) && !isIncreasePartitions(noSqlMigration)) {
            LOGGER.info("Skipping migration {} as it is already done", noSqlMigration);
        } else {
            LOGGER.info("Running migration {}", noSqlMigration);
            try {
                runMigration(noSqlMigration);
                markMigrationAsDone(noSqlMigration);
            } catch (Exception e) {
                throw CarrotException.shouldNotHappenException(new IllegalStateException("Error running database migration " + noSqlMigration.getClassName(), e));
            }
        }
    }

    protected Stream<T> getMigrations() {
        return migrations.stream();
    }
}
