package cn.boboweike.carrot.storage.nosql.mongo;

import cn.boboweike.carrot.CarrotException;
import cn.boboweike.carrot.storage.nosql.common.NoSqlDatabaseCreator;
import cn.boboweike.carrot.storage.nosql.mongo.migrations.*;
import com.mongodb.MongoWriteException;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.FindOneAndReplaceOptions;
import org.bson.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static cn.boboweike.carrot.storage.StorageProviderUtils.*;
import static cn.boboweike.carrot.storage.nosql.mongo.MongoDBPartitionedStorageProvider.toMongoId;
import static com.mongodb.client.model.Filters.eq;
import static java.util.Arrays.asList;

public class MongoDBCreator extends NoSqlDatabaseCreator<MongoMigration> {

    private static final Logger LOGGER = LoggerFactory.getLogger(MongoDBCreator.class);

    private final MongoDatabase carrotDatabase;
    private final String collectionPrefix;
    private final int numberOfPartitions;

    private final MongoCollection<Document> migrationCollection;

    public MongoDBCreator(MongoClient mongoClient, String dbName, int numberOfPartitions) {
        this(mongoClient, dbName, null, numberOfPartitions);
    }

    public MongoDBCreator(MongoClient mongoClient, String dbName, String collectionPrefix, int numberOfPartition) {
        super(Arrays.asList(
                new M001_CreateTaskCollection(),
                new M002_CreateRecurringTaskCollection(),
                new M003_CreateMetadataCollection(),
                new M004_CreateBackgroundTaskServerCollection(),
                new M005_CreateShedLockCollection())
        );
        this.carrotDatabase = mongoClient.getDatabase(dbName);
        this.collectionPrefix = collectionPrefix;
        this.migrationCollection = carrotDatabase.getCollection(elementPrefixer(collectionPrefix, Migrations.NAME));
        this.numberOfPartitions = numberOfPartition;
    }

    public void validateCollections() {
        final List<String> requiredCollectionNames = asList(Tasks.NAME, RecurringTasks.NAME);
        final List<String> availableCollectionNames = carrotDatabase.listCollectionNames().into(new ArrayList<>());
        for (int partition = 0; partition < numberOfPartitions; partition ++) {
            for (String requiredCollectionName : requiredCollectionNames) {
                if (!availableCollectionNames.contains(elementPrefixerWithPartition(collectionPrefix, requiredCollectionName, partition))) {
                    throw new CarrotException("Not all required collections are available by Carrot!");
                }
            }
        }
        if (!availableCollectionNames.contains(elementPrefixer(collectionPrefix, BackgroundTaskServers.NAME)) ||
                !availableCollectionNames.contains(elementPrefixer(collectionPrefix, ShedLock.NAME)) ||
                !availableCollectionNames.contains(elementPrefixer(collectionPrefix, Metadata.NAME))) {
            throw new CarrotException("Not all required collections are available by Carrot!");
        }
    }

    @Override
    protected boolean isIncreasePartitions(MongoMigration mongoMigration) {
        if (isNewMigration(mongoMigration)) return false;
        if (!mongoMigration.supportPartition()) return false;
        int oldNumberOfPartitions = getOldNumberOfPartitions(mongoMigration);
        if (numberOfPartitions > oldNumberOfPartitions) return true;
        if (numberOfPartitions == oldNumberOfPartitions) return false;
        throw new CarrotException(String.format("Carrot does not support decreasing the number of partitions, old = %s, new = %s!", oldNumberOfPartitions, numberOfPartitions));
    }

    private int getOldNumberOfPartitions(MongoMigration mongoMigration) {
        Document document = migrationCollection.find(eq(toMongoId(Migrations.FIELD_ID), mongoMigration.getClassName())).first();
        if (document == null) return -1;
        int oldNumberOfPartitions = document.getInteger(Migrations.FIELD_NUM_OF_PARTITIONS, -1);
        return oldNumberOfPartitions;
    }

    @Override
    protected boolean isNewMigration(MongoMigration mongoMigration) {
        Document document = migrationCollection.find(eq(toMongoId(Migrations.FIELD_ID), mongoMigration.getClassName())).first();
        return document == null;
    }

    @Override
    protected void runMigration(MongoMigration mongoMigration) {
        if (mongoMigration.supportPartition()) {
            boolean increasePartitions = isIncreasePartitions(mongoMigration);
            int oldNumberOfPartitions = getOldNumberOfPartitions(mongoMigration);
            if (increasePartitions) {
                LOGGER.info("Increasing the number of partitions, old {}, new {}", oldNumberOfPartitions, numberOfPartitions);
            }
            for (int partition = 0; partition < numberOfPartitions; partition++) {
                if (increasePartitions && (partition < oldNumberOfPartitions)) continue; // old partitions are already there
                mongoMigration.runMigration(carrotDatabase, collectionPrefix, partition);
            }
        } else {
            mongoMigration.runMigration(carrotDatabase, collectionPrefix, null);
        }
    }

    @Override
    protected boolean markMigrationAsDone(MongoMigration mongoMigration) {
        try {
            Document document = new Document();
            document.put(toMongoId(Migrations.FIELD_ID), mongoMigration.getClassName());
            document.put(Migrations.FIELD_NAME, mongoMigration.getClassName());
            document.put(Migrations.FIELD_DATE, Instant.now());
            if (mongoMigration.supportPartition()) {
                document.put(Migrations.FIELD_NUM_OF_PARTITIONS, numberOfPartitions);
            }
            migrationCollection.findOneAndReplace(eq(toMongoId(Migrations.FIELD_ID), mongoMigration.getClassName()), document, new FindOneAndReplaceOptions().upsert(true));
            return true;
        } catch (MongoWriteException e) {
            if (e.getError().getCode() == 11000) {
                return true;
            }
            throw e;
        }
    }
}
