package cn.boboweike.carrot.storage.nosql.mongo.mapper;

import cn.boboweike.carrot.storage.StorageProviderUtils.*;
import cn.boboweike.carrot.tasks.RecurringTask;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.mappers.TaskMapper;
import cn.boboweike.carrot.tasks.states.ScheduledState;
import cn.boboweike.carrot.tasks.states.StateName;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.UpdateOneModel;
import com.mongodb.client.model.UpdateOptions;
import org.bson.Document;
import org.bson.conversions.Bson;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.UUID;

import static cn.boboweike.carrot.storage.nosql.mongo.MongoDBPartitionedStorageProvider.toMongoId;

public class TaskDocumentMapper {
    private final TaskMapper taskMapper;

    public TaskDocumentMapper(TaskMapper taskMapper) {
        this.taskMapper = taskMapper;
    }

    public Document toInsertDocument(Task task) {
        final Document document = new Document();
        document.put(toMongoId(Tasks.FIELD_ID), task.getId());
        document.put(Tasks.FIELD_VERSION, task.getVersion());
        document.put(Tasks.FIELD_TASK_AS_JSON, taskMapper.serializeTask(task));
        document.put(Tasks.FIELD_TASK_SIGNATURE, task.getTaskSignature());
        document.put(Tasks.FIELD_STATE, task.getState().name());
        document.put(Tasks.FIELD_CREATED_AT, toMicroSeconds(task.getCreatedAt()));
        document.put(Tasks.FIELD_UPDATED_AT, toMicroSeconds(task.getUpdatedAt()));
        if (task.hasState(StateName.SCHEDULED)) {
            document.put(Tasks.FIELD_SCHEDULED_AT, toMicroSeconds(task.<ScheduledState>getTaskState().getScheduledAt()));
        }
        task.getRecurringTaskId().ifPresent(recurringTaskId -> document.put(Tasks.FIELD_RECURRING_TASK_ID, recurringTaskId));
        return document;
    }

    public Document toUpdateDocument(Task task) {
        final Document document = new Document();
        document.put(Tasks.FIELD_VERSION, task.getVersion());
        document.put(Tasks.FIELD_TASK_AS_JSON, taskMapper.serializeTask(task));
        document.put(Tasks.FIELD_STATE, task.getState().name());
        document.put(Tasks.FIELD_UPDATED_AT, toMicroSeconds(task.getUpdatedAt()));
        if (task.hasState(StateName.SCHEDULED)) {
            document.put(Tasks.FIELD_SCHEDULED_AT, toMicroSeconds(((ScheduledState) task.getTaskState()).getScheduledAt()));
        }
        task.getRecurringTaskId().ifPresent(recurringTaskId -> document.put(Tasks.FIELD_RECURRING_TASK_ID, recurringTaskId));
        return new Document("$set", document);
    }

    public UpdateOneModel<Document> toUpdateOneModel(Task task) {
        Document filterDocument = new Document();
        filterDocument.append(toMongoId(Tasks.FIELD_ID), task.getId());
        filterDocument.append(Tasks.FIELD_VERSION, (task.getVersion() -1));

        //Update doc
        Document updateDocument = toUpdateDocument(task);

        //Update option
        UpdateOptions updateOptions = new UpdateOptions();
        updateOptions.upsert(false); //if true, will create a new doc in case of unmatched find

        return new UpdateOneModel<>(filterDocument, updateDocument, updateOptions);
    }

    public Task toTask(Document document) {
        return taskMapper.deserializeTask(document.get(Tasks.FIELD_TASK_AS_JSON).toString());
    }

    public Document toInsertDocument(RecurringTask recurringTask) {
        final Document document = new Document();
        document.put(toMongoId(RecurringTasks.FIELD_ID), recurringTask.getId());
        document.put(RecurringTasks.FIELD_VERSION, recurringTask.getVersion());
        document.put(RecurringTasks.FIELD_TASK_AS_JSON, taskMapper.serializeRecurringTask(recurringTask));
        return document;
    }

    public Bson byId(List<UUID> ids) {
        return Filters.in(toMongoId(Tasks.FIELD_ID), ids);
    }

    public RecurringTask toRecurringTask(Document document) {
        return taskMapper.deserializeRecurringTask(document.get(RecurringTasks.FIELD_TASK_AS_JSON).toString());
    }

    private long toMicroSeconds(Instant instant) {
        return ChronoUnit.MICROS.between(Instant.EPOCH, instant);
    }
}
