package cn.boboweike.carrot.tasks;

import cn.boboweike.carrot.scheduling.Schedule;
import cn.boboweike.carrot.scheduling.ScheduleExpressionType;
import cn.boboweike.carrot.tasks.states.EnqueuedState;
import cn.boboweike.carrot.tasks.states.ScheduledState;

import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Optional;

public class RecurringTask extends AbstractTask {

    private String id;
    private String scheduleExpression;
    private String zoneId;
    private Instant createdAt;

    private RecurringTask() {
        // used for deserialization
    }

    public RecurringTask(String id, TaskDetails taskDetails, String scheduleExpression, String zoneId) {
        this(id, taskDetails, ScheduleExpressionType.getSchedule(scheduleExpression), ZoneId.of(zoneId));
    }

    public RecurringTask(String id, TaskDetails taskDetails, Schedule schedule, ZoneId zoneId) {
        this(id, taskDetails, schedule, zoneId, Instant.now(Clock.system(zoneId)));
    }

    public RecurringTask(String id, TaskDetails taskDetails, String scheduleExpression, String zoneId, String createdAt) {
        this(id, taskDetails, ScheduleExpressionType.getSchedule(scheduleExpression), ZoneId.of(zoneId), Instant.parse(createdAt));
    }

    public RecurringTask(String id, TaskDetails taskDetails, Schedule schedule, ZoneId zoneId, Instant createdAt) {
        super(taskDetails);
        schedule.validateSchedule();
        this.id = validateAndSetId(id);
        this.zoneId = zoneId.getId();
        this.scheduleExpression = schedule.toString();
        this.createdAt = createdAt;
    }

    @Override
    public String getId() {
        return id;
    }

    public String getScheduleExpression() {
        return scheduleExpression;
    }

    public Task toScheduledTask() {
        Instant nextRun = getNextRun();
        final Task task = new Task(getTaskDetails(), new ScheduledState(nextRun, this));
        task.setTaskName(getTaskName());
        task.setRecurringTaskId(getId());
        return task;
    }

    public Task toEnqueuedTask() {
        final Task task = new Task(getTaskDetails(), new EnqueuedState());
        task.setTaskName(getTaskName());
        task.setRecurringTaskId(getId());
        return task;
    }

    public String getZoneId() {
        return zoneId;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public Instant getNextRun() {
        return ScheduleExpressionType
                .getSchedule(scheduleExpression)
                .next(createdAt, ZoneId.of(zoneId));
    }

    private String validateAndSetId(String input) {
        String result = Optional.ofNullable(input).orElse(getTaskSignature().replace("$", "_")); //why: to support inner classes

        if (!result.matches("[\\dA-Za-z-_(),.]+")) {
            throw new IllegalArgumentException("The id of a recurring task can only contain letters and numbers.");
        }
        return result;
    }

    @Override
    public String toString() {
        return "RecurringTask{" +
                "id=" + id +
                ", version='" + getVersion() + '\'' +
                ", identity='" + System.identityHashCode(this) + '\'' +
                ", taskSignature='" + getTaskSignature() + '\'' +
                ", taskName='" + getTaskName() + '\'' +
                '}';
    }
}
