package cn.boboweike.carrot.tasks;

import cn.boboweike.carrot.server.BackgroundTaskServer;
import cn.boboweike.carrot.storage.ConcurrentTaskModificationException;
import cn.boboweike.carrot.tasks.context.TaskDashboardLogger;
import cn.boboweike.carrot.tasks.context.TaskDashboardProgressBar;
import cn.boboweike.carrot.tasks.states.*;
import cn.boboweike.carrot.utils.streams.StreamUtils;

import java.time.Duration;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Stream;

import static cn.boboweike.carrot.storage.PartitionedStorageProvider.PARTITION_HINT_KEY;
import static cn.boboweike.carrot.tasks.states.AllowedTaskStateStateChanges.isIllegalStateChange;
import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.cast;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;

/**
 * Defines the task with its TaskDetails, History and Task Metadata
 */
public class Task extends AbstractTask {

    private final UUID id;
    private final ArrayList<TaskState> taskHistory;
    private final ConcurrentMap<String, Object> metadata;
    private String recurringTaskId;

    private Task() {
        // used for deserialization
        this.id = null;
        this.taskHistory = new ArrayList<>();
        this.metadata = new ConcurrentHashMap<>();
    }

    public Task(TaskDetails taskDetails) {
        this(taskDetails, new EnqueuedState());
    }

    public Task(UUID id, TaskDetails taskDetails) {
        this(id, taskDetails, new EnqueuedState());
    }

    public Task(TaskDetails taskDetails, TaskState taskState) {
        this(null, 0, taskDetails, singletonList(taskState), new ConcurrentHashMap<>());
    }

    public Task(UUID id, TaskDetails taskDetails, TaskState taskState) {
        this(id, 0, taskDetails, singletonList(taskState), new ConcurrentHashMap<>());
    }

    public Task(UUID id, int version, TaskDetails taskDetails, List<TaskState> taskHistory, ConcurrentMap<String, Object> metadata) {
        super(taskDetails, version);
        if (taskHistory.isEmpty()) throw new IllegalStateException("A task should have at least one initial state");
        this.id = id != null ? id : UUID.randomUUID();
        this.taskHistory = new ArrayList<>(taskHistory);
        this.metadata = metadata;
    }

    @Override
    public UUID getId() {
        return id;
    }

    public void setRecurringTaskId(String recurringTaskId) {
        this.recurringTaskId = recurringTaskId;
    }

    public Optional<String> getRecurringTaskId() {
        return Optional.ofNullable(recurringTaskId);
    }

    public List<TaskState> getTaskStates() {
        return unmodifiableList(taskHistory);
    }

    public <T extends TaskState> Stream<T> getTaskStatesOfType(Class<T> clazz) {
        return StreamUtils.ofType(getTaskStates(), clazz);
    }

    public <T extends TaskState> Optional<T> getLastTaskStateOfType(Class<T> clazz) {
        return getTaskStatesOfType(clazz).reduce((first, second) -> second);
    }

    public <T extends TaskState> T getTaskState() {
        return cast(getTaskState(-1));
    }

    public TaskState getTaskState(int element) {
        if (element >= 0) {
            return taskHistory.get(element);
        } else {
            if (Math.abs(element) > taskHistory.size()) return null;
            return taskHistory.get(taskHistory.size() + element);
        }
    }

    public StateName getState() {
        return getTaskState().getName();
    }

    public boolean hasState(StateName state) {
        return getState().equals(state);
    }

    public void enqueue() {
        addTaskState(new EnqueuedState());
    }

    public void scheduleAt(Instant instant, String reason) {
        addTaskState(new ScheduledState(instant, reason));
    }

    public void startProcessingOn(BackgroundTaskServer backgroundTaskServer) {
        if (getState() == StateName.PROCESSING) throw new ConcurrentTaskModificationException(this);
        addTaskState(new ProcessingState(backgroundTaskServer.getId()));
    }

    public void updateProcessing() {
        ProcessingState taskState = getTaskState();
        taskState.setUpdatedAt(Instant.now());
    }

    public void succeeded() {
        Optional<EnqueuedState> lastEnqueuedState = getLastTaskStateOfType(EnqueuedState.class);
        if (!lastEnqueuedState.isPresent()) {
            throw new IllegalStateException("Task cannot succeed if it was not enqueued before.");
        }

        clearMetadata();
        Duration latencyDuration = Duration.between(lastEnqueuedState.get().getEnqueuedAt(), getTaskState().getCreatedAt());
        Duration processDuration = Duration.between(getTaskState().getCreatedAt(), Instant.now());
        addTaskState(new SucceededState(latencyDuration, processDuration));
    }

    public void failed(String message, Exception exception) {
        addTaskState(new FailedState(message, exception));
    }

    public void delete(String reason) {
        clearMetadata();
        addTaskState(new DeletedState(reason));
    }

    public Instant getCreatedAt() {
        return getTaskState(0).getCreatedAt();
    }

    public Instant getUpdatedAt() {
        return getTaskState().getUpdatedAt();
    }

    public Map<String, Object> getMetadata() {
        return metadata;
    }

    @Override
    public String toString() {
        return "Task{" +
                "id=" + id +
                ", version='" + getVersion() + '\'' +
                ", identity='" + System.identityHashCode(this) + '\'' +
                ", taskSignature='" + getTaskSignature() + '\'' +
                ", taskName='" + getTaskName() + '\'' +
                ", taskState='" + getState() + '\'' +
                ", updatedAt='" + getUpdatedAt() + '\'' +
                '}';
    }

    private void addTaskState(TaskState taskState) {
        if (isIllegalStateChange(getState(), taskState.getName())) {
            throw new IllegalTaskStateChangeException(getState(), taskState.getName());
        }
        this.taskHistory.add(taskState);
    }

    private void clearMetadata() {
        metadata.entrySet().removeIf(entry -> !entry.getKey().equals(PARTITION_HINT_KEY) &&
                !(entry.getKey().matches("(\\b" + TaskDashboardLogger.CARROT_LOG_KEY + "\\b|\\b" + TaskDashboardProgressBar.CARROT_PROGRESSBAR_KEY + "\\b)-(\\d)")));
    }
}
