package cn.boboweike.carrot.tasks.context;

import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.states.StateName;

import java.time.Instant;
import java.util.Map;
import java.util.UUID;

import static cn.boboweike.carrot.tasks.context.TaskDashboardLogger.CARROT_LOG_KEY;
import static cn.boboweike.carrot.tasks.context.TaskDashboardProgressBar.CARROT_PROGRESSBAR_KEY;
import static cn.boboweike.carrot.tasks.mappers.MDCMapper.CARROT_MDC_KEY;
import static java.util.Collections.unmodifiableMap;
import static java.util.stream.Collectors.toMap;

/**
 * The TaskContext class gives access to the Task id, the Task name, the state, ... .
 * <p>
 * Using the {@link #getMetadata()}, it also allows to store some data between different task retries so tasks can be made re-entrant.
 * This comes in handy when your task exists out of multiple steps, and you want to keep track of which step already succeeded. Then,
 * in case of a failure, you can skip the steps that already completed successfully.
 * As soon as the task is completed successfully the metadata is cleared (for storage purpose reasons).
 */
public class TaskContext {

    public static final TaskContext Null = new TaskContext(null);

    private final Task task;

    private TaskDashboardLogger taskDashboardLogger;
    private TaskDashboardProgressBar taskDashboardProgressBar;

    protected TaskContext() {
        // Needed for JSON-B deserialization
        this(null);
    }

    /**
     * Keep constructor package protected to remove confusion on how to instantiate the TaskContext.
     * Tip - To use the TaskContext in your task, pass TaskContext.Null
     *
     * @param task the task for this TaskContext
     */
    protected TaskContext(Task task) {
        this.task = task;
    }

    public UUID getTaskId() {
        return task.getId();
    }

    public String getTaskName() {
        return task.getTaskName();
    }

    public StateName getTaskState() {
        return task.getState();
    }

    public Instant getCreatedAt() {
        return task.getCreatedAt();
    }

    public Instant getUpdatedAt() {
        return task.getUpdatedAt();
    }

    public String getSignature() {
        return task.getTaskSignature();
    }

    public TaskDashboardLogger logger() {
        if (taskDashboardLogger == null) {
            taskDashboardLogger = new TaskDashboardLogger(task);
        }
        return taskDashboardLogger;
    }

    public TaskDashboardProgressBar progressBar(int totalAmount) {
        return progressBar((long) totalAmount);
    }

    public TaskDashboardProgressBar progressBar(long totalAmount) {
        if (taskDashboardProgressBar == null) {
            taskDashboardProgressBar = new TaskDashboardProgressBar(task, totalAmount);
        }
        return taskDashboardProgressBar;
    }

    /**
     * Gives access to Task Metadata via an UnmodifiableMap. To save Metadata, use the {@link #saveMetadata(String, Object)} method
     *
     * @return all user defined metadata about a Task. This metadata is only accessible up to the point a task succeeds.
     */
    public Map<String, Object> getMetadata() {
        return unmodifiableMap(
                task.getMetadata().entrySet().stream()
                        .filter(entry -> !entry.getKey().startsWith(CARROT_LOG_KEY))
                        .filter(entry -> !entry.getKey().startsWith(CARROT_PROGRESSBAR_KEY))
                        .filter(entry -> !entry.getKey().startsWith(CARROT_MDC_KEY))
                        .collect(toMap(Map.Entry::getKey, Map.Entry::getValue))
        );
    }

    /**
     * Allows saving metadata for a certain Task. The value must either be a simple type (String, UUID, Integers, ...) or implement the Metadata interface for serialization to Json.
     * Note that it is important that the objects you save are <b>thread-safe</b> (e.g. a CopyOnWriteArrayList, ... ).
     * <p>
     * If the key already exists, the metadata is updated.
     *
     * @param key      the key to store the metadata
     * @param metadata the metadata itself
     */
    public void saveMetadata(String key, Object metadata) {
        validateMetadata(metadata);
        task.getMetadata().put(key, metadata);
    }

    /**
     * Allows saving metadata for a certain task. The value must either be a simple type (String, UUID, Integers, ...) or implement the Metadata interface for serialization to Json.
     * Note that it is important that the objects you save are <b>thread-safe</b> (e.g. a CopyOnWriteArrayList, ... ).
     * <p>
     * If the key already exists, the metadata is NOT updated.
     *
     * @param key      the key to store the metadata
     * @param metadata the metadata itself
     */
    public void saveMetadataIfAbsent(String key, Object metadata) {
        validateMetadata(metadata);
        task.getMetadata().putIfAbsent(key, metadata);
    }

    private static void validateMetadata(Object metadata) {
        if (!(metadata.getClass().getName().startsWith("java.") || metadata instanceof Metadata)) {
            throw new IllegalArgumentException("All task metadata must either be a simple type (String, UUID, Integers, ...) or implement the Metadata interface for serialization to Json.");
        }
    }

    // marker interface for Json Serialization
    public interface Metadata {

    }
}

