package cn.boboweike.carrot.tasks.context;

import cn.boboweike.carrot.tasks.Task;

import java.time.Instant;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.cast;

public class TaskDashboardLogger {

    public enum Level {
        INFO, WARN, ERROR
    }

    public static final String CARROT_LOG_KEY = "carrotDashboardLog";

    private final TaskDashboardLogLines logLines;

    public TaskDashboardLogger(Task task) {
        this.logLines = initLogLines(task);
    }

    public void info(String infoMessage) {
        logLines.add(new TaskDashboardLogLine(Level.INFO, infoMessage));
    }

    public void warn(String warnMessage) {
        logLines.add(new TaskDashboardLogLine(Level.WARN, warnMessage));
    }

    public void error(String errorMessage) {
        logLines.add(new TaskDashboardLogLine(Level.ERROR, errorMessage));
    }

    private TaskDashboardLogLines initLogLines(Task task) {
        Map<String, Object> taskMetadata = task.getMetadata();
        String logKey = logKey(task.getTaskStates().size());
        taskMetadata.putIfAbsent(logKey, new TaskDashboardLogLines());
        return cast(taskMetadata.get(logKey));
    }

    /**
     * Returns a unique key based on the current taskState (so that all logs regarding the first processing attempt can be displayed under the first processing view in the dashboard, ... )
     *
     * @param taskStateNbr the current state nbr - typically enqueued=1, processing=2, failed=3, scheduled=4, enqueued=5, processing=6, ...
     * @return a log key for the metadata matching the current task state.
     */
    private static String logKey(int taskStateNbr) {
        return CARROT_LOG_KEY + "-" + taskStateNbr;
    }

    public static class TaskDashboardLogLines implements TaskContext.Metadata {

        /* Must be data structure that can be serialized to json and that allows iteration while being updated */
        private ConcurrentLinkedQueue<TaskDashboardLogLine> logLines;

        public TaskDashboardLogLines() {
            this.logLines = new ConcurrentLinkedQueue<>();
        }

        public void add(TaskDashboardLogLine line) {
            logLines.add(line);
        }

        public Queue<TaskDashboardLogLine> getLogLines() {
            return logLines;
        }
    }

    public static class TaskDashboardLogLine {

        private Level level;
        private Instant logInstant;
        private String logMessage;

        protected TaskDashboardLogLine() {
            // for json deserialization
        }

        public TaskDashboardLogLine(Level level, String logMessage) {
            this.level = level;
            this.logInstant = Instant.now();
            this.logMessage = logMessage;
        }

        public Level getLevel() {
            return level;
        }

        public Instant getLogInstant() {
            return logInstant;
        }

        public String getLogMessage() {
            return logMessage;
        }
    }
}