package cn.boboweike.carrot.tasks.context;

import cn.boboweike.carrot.tasks.Task;

import java.util.Map;

import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.cast;

public class TaskDashboardProgressBar {

    public static final String CARROT_PROGRESSBAR_KEY = "carrotDashboardProgressBar";

    private final TaskDashboardProgress taskDashboardProgress;

    public TaskDashboardProgressBar(Task task, Long totalAmount) {
        this.taskDashboardProgress = initTaskDashboardProgress(task, totalAmount);
    }

    private TaskDashboardProgress initTaskDashboardProgress(Task task, Long totalAmount) {
        Map<String, Object> taskMetadata = task.getMetadata();
        String progressBarKey = progressBarKey(task.getTaskStates().size());
        taskMetadata.putIfAbsent(progressBarKey, new TaskDashboardProgress(totalAmount));
        return cast(taskMetadata.get(progressBarKey));
    }

    public void increaseByOne() {
        taskDashboardProgress.increaseByOne();
    }

    public int getProgress() {
        return taskDashboardProgress.getProgress();
    }

    public void setValue(int currentProgress) {
        this.setValue((long) currentProgress);
    }

    public void setValue(long currentProgress) {
        this.taskDashboardProgress.setCurrentValue(currentProgress);
    }

    /**
     * Returns a unique key based on the current taskState (so that the progressbar regarding the first processing attempt can be displayed under the first processing view in the dashboard, ... )
     *
     * @param taskStateNbr the current state nbr - typically enqueued=1, processing=2, failed=3, scheduled=4, enqueued=5, processing=6, ...
     * @return a progress bar key for the metadata matching the current task state.
     */
    private static String progressBarKey(int taskStateNbr) {
        return CARROT_PROGRESSBAR_KEY + "-" + taskStateNbr;
    }

    private static class TaskDashboardProgress implements TaskContext.Metadata {

        private Long totalAmount;
        private Long currentValue;
        private int progress;

        protected TaskDashboardProgress() {
            // for json deserialization
        }

        public TaskDashboardProgress(Long totalAmount) {
            if (totalAmount < 1) throw new IllegalArgumentException("The total progress amount must be larger than 0.");
            this.totalAmount = totalAmount;
            this.currentValue = 0L;
        }

        public void increaseByOne() {
            setCurrentValue(currentValue + 1);
        }

        public void setCurrentValue(Long currentValue) {
            this.currentValue = currentValue;
            this.progress = (int) (currentValue * 100 / totalAmount);
        }

        public int getProgress() {
            return progress;
        }
    }
}

