package cn.boboweike.carrot.tasks.details;

import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.details.instructions.*;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.*;

import java.io.IOException;
import java.io.InputStream;

import static cn.boboweike.carrot.CarrotException.shouldNotHappenException;


abstract class AbstractTaskDetailsFinder extends ClassVisitor {

    protected final TaskDetailsBuilder taskDetailsBuilder;

    protected AbstractTaskDetailsFinder(TaskDetailsBuilder taskDetailsBuilder) {
        super(Opcodes.ASM7);
        this.taskDetailsBuilder = taskDetailsBuilder;
    }

    @Override
    public MethodVisitor visitMethod(int access, String name, String descriptor, String signature, String[] exceptions) {
        if (isLambdaContainingTaskDetails(name)) {
            return new MethodVisitor(Opcodes.ASM7) {

                @Override
                public void visitFieldInsn(int opcode, String owner, String name, String descriptor) {
                    VisitFieldInstruction instruction = AllJVMInstructions.get(opcode, taskDetailsBuilder);
                    instruction.load(owner, name, descriptor);
                }

                @Override
                public void visitInvokeDynamicInsn(String name, String descriptor, Handle bootstrapMethodHandle, Object... bootstrapMethodArguments) {
                    InvokeDynamicInstruction instruction = AllJVMInstructions.get(Opcodes.INVOKEDYNAMIC, taskDetailsBuilder);
                    instruction.load(name, descriptor, bootstrapMethodHandle, bootstrapMethodArguments);
                }

                @Override
                public void visitMethodInsn(int opcode, String owner, String name, String descriptor, boolean isInterface) {
                    VisitMethodInstruction visitMethodInstruction = AllJVMInstructions.get(opcode, taskDetailsBuilder);
                    visitMethodInstruction.load(owner, name, descriptor, isInterface);
                }

                @Override
                public void visitInsn(int opcode) {
                    ZeroOperandInstruction zeroOperandInstruction = AllJVMInstructions.get(opcode, taskDetailsBuilder);
                    zeroOperandInstruction.load();
                }

                @Override
                public void visitVarInsn(int opcode, int variable) {
                    VisitLocalVariableInstruction instruction = AllJVMInstructions.get(opcode, taskDetailsBuilder);
                    instruction.load(variable);
                }

                @Override
                public void visitIntInsn(int opcode, int operand) {
                    SingleIntOperandInstruction singleIntOperandInstruction = AllJVMInstructions.get(opcode, taskDetailsBuilder);
                    singleIntOperandInstruction.load(operand);
                }

                @Override
                public void visitLdcInsn(Object value) {
                    LdcInstruction ldcInstruction = AllJVMInstructions.get(Opcodes.LDC, taskDetailsBuilder);
                    ldcInstruction.load(value);
                }

                @Override
                public void visitTypeInsn(int opcode, String type) {
                    VisitTypeInstruction instruction = AllJVMInstructions.get(opcode, taskDetailsBuilder);
                    instruction.load(type);
                }
            };
        } else {
            return super.visitMethod(access, name, descriptor, signature, exceptions);
        }
    }

    protected abstract boolean isLambdaContainingTaskDetails(String name);

    protected abstract InputStream getClassContainingLambdaAsInputStream();

    public TaskDetails getTaskDetails() {
        return taskDetailsBuilder.getTaskDetails();
    }

    protected void parse(InputStream inputStream) {
        try {
            ClassReader parser = new ClassReader(inputStream);
            parser.accept(this, ClassReader.SKIP_FRAMES);
        } catch (IOException e) {
            throw shouldNotHappenException(e);
        }
    }

}
