package cn.boboweike.carrot.tasks.details;

import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.TaskParameter;
import cn.boboweike.carrot.tasks.lambdas.*;
import cn.boboweike.carrot.utils.reflection.ReflectionUtils;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

import static cn.boboweike.carrot.CarrotException.shouldNotHappenException;
import static java.lang.Boolean.TRUE;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;

public class CachingTaskDetailsGenerator implements TaskDetailsGenerator {

    private final TaskDetailsGenerator delegate;
    private final Map<Class<?>, CacheableTaskDetails> cache;

    public CachingTaskDetailsGenerator() {
        this(new TaskDetailsAsmGenerator());
    }

    public CachingTaskDetailsGenerator(TaskDetailsGenerator delegate) {
        this.delegate = delegate;
        this.cache = new ConcurrentHashMap<>();
    }

    @Override
    public TaskDetails toTaskDetails(TaskLambda lambda) {
        cache.computeIfAbsent(lambda.getClass(), clazz -> new CacheableTaskDetails(delegate));
        return cache.get(lambda.getClass()).getTaskDetails(lambda);
    }

    @Override
    public TaskDetails toTaskDetails(IocTaskLambda<?> lambda) {
        cache.computeIfAbsent(lambda.getClass(), clazz -> new CacheableTaskDetails(delegate));
        return cache.get(lambda.getClass()).getTaskDetails(lambda);
    }

    @Override
    public <T> TaskDetails toTaskDetails(T itemFromStream, TaskLambdaFromStream<T> lambda) {
        cache.computeIfAbsent(lambda.getClass(), clazz -> new CacheableTaskDetails(delegate));
        return cache.get(lambda.getClass()).getTaskDetails(itemFromStream, lambda);
    }

    @Override
    public <S, T> TaskDetails toTaskDetails(T itemFromStream, IocTaskLambdaFromStream<S, T> lambda) {
        cache.computeIfAbsent(lambda.getClass(), clazz -> new CacheableTaskDetails(delegate));
        return cache.get(lambda.getClass()).getTaskDetails(itemFromStream, lambda);
    }

    private static class CacheableTaskDetails {

        private static final MethodHandles.Lookup lookup = MethodHandles.lookup();
        private final TaskDetailsGenerator taskDetailsGeneratorDelegate;
        private final ReentrantLock taskDetailsLock;
        private TaskDetails taskDetails;
        private List<TaskParameterRetriever> taskParameterRetrievers;

        private CacheableTaskDetails(TaskDetailsGenerator taskDetailsGeneratorDelegate) {
            this.taskDetailsGeneratorDelegate = taskDetailsGeneratorDelegate;
            this.taskDetailsLock = new ReentrantLock();
        }

        public TaskDetails getTaskDetails(TaskLambda lambda) {
            if (taskDetails == null) {
                taskDetailsLock.lock();
                try {
                    taskDetails = taskDetailsGeneratorDelegate.toTaskDetails(lambda);
                    taskParameterRetrievers = initTaskParameterRetrievers(taskDetails, lambda, Optional.empty());
                    return taskDetails;
                } finally {
                    taskDetailsLock.unlock();
                }
            } else if (TRUE.equals(taskDetails.getCacheable())) {
                return getCachedTaskDetails(lambda, Optional.empty());
            } else {
                return taskDetailsGeneratorDelegate.toTaskDetails(lambda);
            }
        }

        public TaskDetails getTaskDetails(IocTaskLambda lambda) {
            if (taskDetails == null) {
                taskDetailsLock.lock();
                try {
                    taskDetails = taskDetailsGeneratorDelegate.toTaskDetails(lambda);
                    taskParameterRetrievers = initTaskParameterRetrievers(taskDetails, lambda, Optional.empty());
                    return taskDetails;
                } finally {
                    taskDetailsLock.unlock();
                }
            } else if (TRUE.equals(taskDetails.getCacheable())) {
                return getCachedTaskDetails(lambda, Optional.empty());
            } else {
                return taskDetailsGeneratorDelegate.toTaskDetails(lambda);
            }
        }

        public <T> TaskDetails getTaskDetails(T itemFromStream, TaskLambdaFromStream<T> lambda) {
            if (taskDetails == null) {
                taskDetailsLock.lock();
                try {
                    taskDetails = taskDetailsGeneratorDelegate.toTaskDetails(itemFromStream, lambda);
                    taskParameterRetrievers = initTaskParameterRetrievers(taskDetails, lambda, Optional.of(itemFromStream));
                    return taskDetails;
                } finally {
                    taskDetailsLock.unlock();
                }
            } else if (TRUE.equals(taskDetails.getCacheable())) {
                return getCachedTaskDetails(lambda, Optional.of(itemFromStream));
            } else {
                return taskDetailsGeneratorDelegate.toTaskDetails(itemFromStream, lambda);
            }
        }

        public <S, T> TaskDetails getTaskDetails(T itemFromStream, IocTaskLambdaFromStream<S, T> lambda) {
            if (taskDetails == null) {
                taskDetailsLock.lock();
                try {
                    taskDetails = taskDetailsGeneratorDelegate.toTaskDetails(itemFromStream, lambda);
                    taskParameterRetrievers = initTaskParameterRetrievers(taskDetails, lambda, Optional.of(itemFromStream));
                    return taskDetails;
                } finally {
                    taskDetailsLock.unlock();
                }
            } else if (TRUE.equals(taskDetails.getCacheable())) {
                return getCachedTaskDetails(lambda, Optional.of(itemFromStream));
            } else {
                return taskDetailsGeneratorDelegate.toTaskDetails(itemFromStream, lambda);
            }
        }

        private static <T> List<TaskParameterRetriever> initTaskParameterRetrievers(TaskDetails taskDetails, CarrotTask carrotTask, Optional<T> itemFromStream) {
            try {
                List<TaskParameterRetriever> parameterRetrievers = new ArrayList<>();
                List<Field> declaredFields = new ArrayList<>(asList(carrotTask.getClass().getDeclaredFields()));
                List<TaskParameter> taskParameters = taskDetails.getTaskParameters();

                if (!declaredFields.isEmpty()
                        && !(declaredFields.get(0).getType().getName().startsWith("java."))
                        && (carrotTask instanceof TaskLambda || carrotTask instanceof TaskLambdaFromStream)) {
                    declaredFields.remove(0);
                }

                for (TaskParameter tp : taskParameters) {
                    parameterRetrievers.add(createTaskParameterRetriever(tp, carrotTask, itemFromStream, declaredFields));
                }

                taskDetails.setCacheable(declaredFields.isEmpty() && taskParameters.size() == parameterRetrievers.size());
                return parameterRetrievers;
            } catch (Exception e) {
                taskDetails.setCacheable(false);
                return emptyList();
            }
        }

        private static <T> TaskParameterRetriever createTaskParameterRetriever(TaskParameter tp, CarrotTask carrotTask, Optional<T> itemFromStream, List<Field> declaredFields) throws IllegalAccessException {
            TaskParameterRetriever taskParameterRetriever = new FixedTaskParameterRetriever(tp);
            if (itemFromStream.isPresent() && tp.getObject().equals(itemFromStream.get())) {
                taskParameterRetriever = new ItemFromStreamTaskParameterRetriever(tp);
            } else {
                final ListIterator<Field> fieldIterator = declaredFields.listIterator();
                while (fieldIterator.hasNext()) {
                    Field f = fieldIterator.next();
                    Object valueFromField = ReflectionUtils.getValueFromField(f, carrotTask);
                    if (tp.getObject().equals(valueFromField)) {
                        MethodHandle e = lookup.unreflectGetter(f);
                        taskParameterRetriever = new MethodHandleTaskParameterRetriever(tp, e.asType(e.type().generic()));
                        fieldIterator.remove();
                        break;
                    }
                }
            }
            return taskParameterRetriever;
        }

        private <T> TaskDetails getCachedTaskDetails(CarrotTask task, Optional<T> itemFromStream) {
            final TaskDetails cachedTaskDetails = new TaskDetails(
                    this.taskDetails.getClassName(),
                    this.taskDetails.getStaticFieldName(),
                    this.taskDetails.getMethodName(),
                    taskParameterRetrievers.stream()
                            .map(taskParameterRetriever -> taskParameterRetriever.getTaskParameter(task, itemFromStream))
                            .collect(Collectors.toList())
            );
            cachedTaskDetails.setCacheable(true);
            return cachedTaskDetails;
        }
    }

    private interface TaskParameterRetriever {
        <T> TaskParameter getTaskParameter(CarrotTask task, Optional<T> itemFromStream);
    }

    private static class FixedTaskParameterRetriever implements TaskParameterRetriever {

        private final TaskParameter taskParameter;

        public FixedTaskParameterRetriever(TaskParameter taskParameter) {
            this.taskParameter = taskParameter;
        }

        @Override
        public <T> TaskParameter getTaskParameter(CarrotTask task, Optional<T> itemFromStream) {
            return taskParameter;
        }
    }

    private static class MethodHandleTaskParameterRetriever implements TaskParameterRetriever {

        private final String taskParameterClassName;
        private final MethodHandle methodHandle;

        public MethodHandleTaskParameterRetriever(TaskParameter taskParameter, MethodHandle methodHandle) {
            this.taskParameterClassName = taskParameter.getClassName();
            this.methodHandle = methodHandle;
        }

        @Override
        public <T> TaskParameter getTaskParameter(CarrotTask task, Optional<T> itemFromStream) {
            try {
                Object o = (Object) methodHandle.invokeExact((Object) task);
                return new TaskParameter(taskParameterClassName, o);
            } catch (Throwable throwable) {
                throw shouldNotHappenException(throwable);
            }
        }
    }

    private static class ItemFromStreamTaskParameterRetriever implements TaskParameterRetriever {

        private final String taskParameterClassName;

        public ItemFromStreamTaskParameterRetriever(TaskParameter taskParameter) {
            this.taskParameterClassName = taskParameter.getClassName();
        }

        @Override
        public <T> TaskParameter getTaskParameter(CarrotTask task, Optional<T> itemFromStream) {
            return new TaskParameter(taskParameterClassName, itemFromStream.orElseThrow(() -> shouldNotHappenException("Can not find itemFromStream")));
        }
    }
}

