package cn.boboweike.carrot.tasks.details;

import cn.boboweike.carrot.tasks.lambdas.CarrotTask;
import cn.boboweike.carrot.utils.reflection.ReflectionUtils;
import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.Opcodes;

import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.Optional;

import static cn.boboweike.carrot.tasks.details.TaskDetailsGeneratorUtils.getKotlinClassContainingLambdaAsInputStream;
import static cn.boboweike.carrot.tasks.details.TaskDetailsGeneratorUtils.toFQResource;
import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.cast;
import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.getValueFromField;

public class KotlinTaskDetailsFinder extends AbstractTaskDetailsFinder {

    private static final String INVOKE = "invoke";
    private static final String ACCEPT = "accept";
    private static final String RUN = "run";

    private enum KotlinVersion {
        ONE_FOUR,
        ONE_FIVE,
        ONE_SIX
    }

    private final CarrotTask carrotTask;
    private int methodCounter = 0;
    private KotlinVersion kotlinVersion;

    private String nestedKotlinClassWithMethodReference;

    KotlinTaskDetailsFinder(CarrotTask carrotTask, Object... params) {
        super(new KotlinTaskDetailsBuilder(carrotTask, params));
        this.carrotTask = carrotTask;
        parse(getClassContainingLambdaAsInputStream());
    }

    @Override
    public AnnotationVisitor visitAnnotation(String descriptor, boolean visible) {
        return new AnnotationVisitor(Opcodes.ASM7) {
            @Override
            public void visit(String name, Object value) {
                if ("mv".equals(name)) {
                    int[] version = cast(value);
                    if (version[0] == 1 && version[1] == 4) {
                        kotlinVersion = KotlinVersion.ONE_FOUR;
                    } else if (version[0] == 1 && version[1] == 5) {
                        kotlinVersion = KotlinVersion.ONE_FIVE;
                    } else if (version[0] == 1 && version[1] == 6) {
                        kotlinVersion = KotlinVersion.ONE_SIX;
                    } else {
                        throw new UnsupportedOperationException("The Kotlin version " + version[0] + "." + version[1] + " is unsupported");
                    }
                }
            }
        };
    }

    @Override
    protected boolean isLambdaContainingTaskDetails(String name) {
        if (name.equals(ACCEPT) || name.equals(INVOKE)) {
            methodCounter++;
        }
        if (KotlinVersion.ONE_FOUR.equals(kotlinVersion)) {
            return name.equals(RUN) || ((name.equals(ACCEPT) || name.equals(INVOKE)) && methodCounter == 2);
        } else {
            return name.equals(RUN) || ((name.equals(ACCEPT) || name.equals(INVOKE)) && methodCounter == 1);
        }
    }

    @Override
    public void visitInnerClass(String name, String outerName, String innerName, int access) {
        if (access == 0x1018 || access == 0x1000) {
            this.nestedKotlinClassWithMethodReference = name;
        }
    }

    @Override
    protected InputStream getClassContainingLambdaAsInputStream() {
        return getKotlinClassContainingLambdaAsInputStream(carrotTask);
    }

    @Override
    protected void parse(InputStream inputStream) {
        Optional<Field> field = ReflectionUtils.findField(carrotTask.getClass(), "function");
        if (field.isPresent()) {
            getTaskDetailsFromKotlinFunction(field.get());
        } else {
            super.parse(inputStream);
            parseNestedClassIfItIsAMethodReference();
        }
    }

    private void getTaskDetailsFromKotlinFunction(Field field) {
        Object function = getValueFromField(field, carrotTask);
        Field receiver = ReflectionUtils.getField(function.getClass(), "receiver");
        Field name = ReflectionUtils.getField(function.getClass(), "name");
        Class<?> receiverClass = getValueFromField(receiver, function).getClass();
        String methodName = cast(getValueFromField(name, function));
        taskDetailsBuilder.setClassName(receiverClass.getName());
        taskDetailsBuilder.setMethodName(methodName);
    }

    private void parseNestedClassIfItIsAMethodReference() {
        boolean isNestedKotlinClassWithMethodReference = nestedKotlinClassWithMethodReference != null
                && !toFQResource(carrotTask.getClass().getName()).equals(nestedKotlinClassWithMethodReference);

        if (isNestedKotlinClassWithMethodReference) {
            String location = "/" + nestedKotlinClassWithMethodReference + ".class";
            super.parse(carrotTask.getClass().getResourceAsStream(location));
            while (taskDetailsBuilder.getInstructions().size() > 1) {
                taskDetailsBuilder.pollFirstInstruction();
            }
        }
    }

}
