package cn.boboweike.carrot.tasks.details;

import cn.boboweike.carrot.CarrotException;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.TaskParameter;
import cn.boboweike.carrot.tasks.details.instructions.AbstractJVMInstruction;
import cn.boboweike.carrot.tasks.details.postprocess.CGLibPostProcessor;
import cn.boboweike.carrot.tasks.details.postprocess.TaskDetailsPostProcessor;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import static java.util.Collections.singletonList;

public abstract class TaskDetailsBuilder {

    private final LinkedList<AbstractJVMInstruction> instructions;
    private final LinkedList<Object> stack;
    private final List<Object> localVariables;
    private String taskDetailsClassName;
    private String taskDetailsStaticFieldName;
    private String taskDetailsMethodName;
    private List<TaskParameter> taskDetailsTaskParameters;
    private List<TaskDetailsPostProcessor> taskDetailsPostProcessors;

    protected TaskDetailsBuilder(List<Object> localVariables) {
        this(localVariables, null, null);
    }

    protected TaskDetailsBuilder(List<Object> localVariables, String className, String methodName) {
        this.instructions = new LinkedList<>();
        this.stack = new LinkedList<>();
        this.localVariables = localVariables;

        setClassName(className);
        setMethodName(methodName);
        setTaskParameters(new ArrayList<>());
        taskDetailsPostProcessors = singletonList(new CGLibPostProcessor());
    }

    public void pushInstructionOnStack(AbstractJVMInstruction jvmInstruction) {
        instructions.add(jvmInstruction);
    }

    public Object getLocalVariable(int nbrInStack) {
        if (nbrInStack < localVariables.size()) {
            return localVariables.get(nbrInStack);
        }
        throw CarrotException.shouldNotHappenException("Can not find variable " + nbrInStack + " in stack");
    }

    public void addLocalVariable(Object o) {
        this.localVariables.add(o);
    }

    public List<AbstractJVMInstruction> getInstructions() {
        return instructions;
    }

    public AbstractJVMInstruction pollFirstInstruction() {
        return instructions.pollFirst();
    }

    public LinkedList<Object> getStack() {
        return stack;
    }

    public TaskDetails getTaskDetails() {
        invokeInstructions();
        final TaskDetails taskDetails = new TaskDetails(taskDetailsClassName, taskDetailsStaticFieldName, taskDetailsMethodName, taskDetailsTaskParameters);
        return postProcessTaskDetails(taskDetails);
    }

    private TaskDetails postProcessTaskDetails(TaskDetails taskDetails) {
        TaskDetails currentTaskDetails = taskDetails;
        for (TaskDetailsPostProcessor postProcessor : getTaskDetailsPostProcessors()) {
            currentTaskDetails = postProcessor.postProcess(currentTaskDetails);
        }
        return currentTaskDetails;
    }

    private void invokeInstructions() {
        if (instructions.isEmpty() && localVariables.size() > 1) { // it is a method reference
            for (int i = 1; i < localVariables.size(); i++) {
                Object variable = localVariables.get(i);
                taskDetailsTaskParameters.add(new TaskParameter(variable));
            }
        } else {
            AbstractJVMInstruction instruction = pollFirstInstruction();
            while (instruction != null) {
                instruction.invokeInstructionAndPushOnStack();
                instruction = pollFirstInstruction();
            }
        }
    }

    public void setClassName(String className) {
        if (taskDetailsStaticFieldName == null) {
            taskDetailsClassName = className;
        }
    }

    public void setStaticFieldName(String name) {
        taskDetailsStaticFieldName = name;
    }

    public void setMethodName(String name) {
        taskDetailsMethodName = name;
    }

    public void setTaskParameters(List<TaskParameter> taskParameters) {
        taskDetailsTaskParameters = taskParameters;
    }

    List<TaskDetailsPostProcessor> getTaskDetailsPostProcessors() {
        return taskDetailsPostProcessors;
    }
}
