package cn.boboweike.carrot.tasks.details.instructions;

import cn.boboweike.carrot.tasks.TaskParameter;
import cn.boboweike.carrot.tasks.details.TaskDetailsBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Stream;

import static cn.boboweike.carrot.CarrotException.shouldNotHappenException;
import static cn.boboweike.carrot.tasks.details.TaskDetailsGeneratorUtils.*;
import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.isClassAssignableToObject;
import static java.util.stream.Collectors.joining;

public class InvokeStaticInstruction extends VisitMethodInstruction {

    private static final Logger LOGGER = LoggerFactory.getLogger(InvokeStaticInstruction.class);

    public InvokeStaticInstruction(TaskDetailsBuilder taskDetailsBuilder) {
        super(taskDetailsBuilder);
    }

    @Override
    public Object invokeInstruction() {
        if (isLastInstruction()) {
            taskDetailsBuilder.setClassName(getClassName());
            taskDetailsBuilder.setMethodName(getMethodName());
            taskDetailsBuilder.setTaskParameters(getTaskParameters());
            return null;
        } else {
            final long before = System.nanoTime();
            final Object result = getObject();
            final long after = System.nanoTime();
            if ((after - before) > 3_000_000) {
                LOGGER.warn("You are using a custom method ({}.{}({})) while enqueueing that takes a lot of time. See https://www.boboweike.cn/carrot/documentation/background-methods/best-practices/ on how to use Carrot effectively.", getClassName(), getMethodName(), Stream.of(findParamTypesFromDescriptorAsArray(descriptor)).map(Class::getSimpleName).collect(joining(", ")));
            }
            return isVoidInstruction() ? AbstractJVMInstruction.DO_NOT_PUT_ON_STACK : result;
        }
    }

    private Object getObject() {
        Class<?>[] paramTypes = findParamTypesFromDescriptorAsArray(descriptor);
        List<Object> parameters = getParametersUsingParamTypes(paramTypes);
        if(isKotlinNullCheck()) return null;
        Object result = createObjectViaStaticMethod(getClassName(), getMethodName(), paramTypes, parameters.toArray());
        return result;
    }

    private boolean isKotlinNullCheck() {
        return getClassName().startsWith("kotlin.") && getMethodName().equals("checkNotNullParameter");
    }

    private String getMethodName() {
        return name;
    }

    private String getClassName() {
        return toFQClassName(owner);
    }

    private List<TaskParameter> getTaskParameters() {
        final List<Class<?>> paramTypesFromDescriptor = findParamTypesFromDescriptor(descriptor);
        final LinkedList<Class<?>> paramTypes = new LinkedList<>(paramTypesFromDescriptor);

        List<TaskParameter> result = new ArrayList<>();
        while (!paramTypes.isEmpty()) {
            result.add(0, toTaskParameter(paramTypes.pollLast(), taskDetailsBuilder.getStack().pollLast()));
        }
        return result;
    }

    private TaskParameter toTaskParameter(Class<?> paramType, Object param) {
        if (isClassAssignableToObject(paramType, param)) {
            if (boolean.class.equals(paramType) && Integer.class.equals(param.getClass()))
                return new TaskParameter(paramType, ((Integer) param) > 0);
            return new TaskParameter(paramType, param);
        } else {
            throw shouldNotHappenException(new IllegalStateException("The found parameter types do not match the parameters."));
        }
    }
}
