package cn.boboweike.carrot.tasks.details.instructions;

import cn.boboweike.carrot.CarrotException;
import cn.boboweike.carrot.tasks.TaskParameter;
import cn.boboweike.carrot.tasks.details.TaskDetailsBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static cn.boboweike.carrot.CarrotException.shouldNotHappenException;
import static cn.boboweike.carrot.tasks.details.TaskDetailsGeneratorUtils.*;
import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.*;
import static java.util.stream.Collectors.joining;

public class TaskDetailsInstruction extends VisitMethodInstruction {

    private static final Logger LOGGER = LoggerFactory.getLogger(TaskDetailsInstruction.class);

    public TaskDetailsInstruction(TaskDetailsBuilder taskDetailsBuilder) {
        super(taskDetailsBuilder);
    }

    @Override
    public Object invokeInstruction() {
        if (!isLastInstruction() && isVoidInstruction()) {
            throw new CarrotException("Carrot only supports enqueueing/scheduling of one method");
        } else if (isLastInstruction()) {
            taskDetailsBuilder.setClassName(getClassName());
            taskDetailsBuilder.setMethodName(getMethodName());
            taskDetailsBuilder.setTaskParameters(getTaskParameters());
            return null;
        } else if (owner.startsWith("java")) {
            return getObject();
        } else {
            final long before = System.nanoTime();
            final Object result = getObject();
            final long after = System.nanoTime();
            if ((after - before) > 3_000_000) {
                LOGGER.warn("You are using a custom method ({}.{}({})) while enqueueing that takes a lot of time. See https://www.boboweike.cn/carrot/documentation/background-methods/best-practices/ on how to use Carrot effectively.", getClassName(), getMethodName(), Stream.of(findParamTypesFromDescriptorAsArray(descriptor)).map(Class::getSimpleName).collect(joining(", ")));
            }
            return result;
        }
    }

    String getClassName() {
        String className = toFQClassName(owner);
        if (taskDetailsBuilder.getStack().isEmpty()) {
            return findInheritedClassName(className).orElse(className);
        }

        Object taskOnStack = taskDetailsBuilder.getStack().getLast();
        if (taskOnStack == null) {
            return className;
        }

        Class<Object> taskClass = toClass(className);
        if (taskClass.isAssignableFrom(taskOnStack.getClass())) {
            return taskOnStack.getClass().getName();
        }
        return className;
    }


    String getMethodName() {
        return name;
    }

    private Object getObject() {
        Class<?>[] paramTypes = findParamTypesFromDescriptorAsArray(descriptor);
        final Object ownerObject = taskDetailsBuilder.getStack().remove(taskDetailsBuilder.getStack().size() - 1 - paramTypes.length);
        return createObjectViaMethod(ownerObject, name, paramTypes, getParametersUsingParamTypes(paramTypes).toArray());
    }

    private Optional<String> findInheritedClassName(String className) {
        if (taskDetailsBuilder.getLocalVariable(0) != null) {
            final Field declaredField = taskDetailsBuilder.getLocalVariable(0).getClass().getDeclaredFields()[0];
            final Object valueFromField = getValueFromField(declaredField, taskDetailsBuilder.getLocalVariable(0));
            if (toClass(className).isAssignableFrom(valueFromField.getClass())) {
                return Optional.of(valueFromField.getClass().getName());
            }
        }
        return Optional.empty();
    }

    private List<TaskParameter> getTaskParameters() {
        final List<Class<?>> paramTypesFromDescriptor = findParamTypesFromDescriptor(descriptor);
        final LinkedList<Class<?>> paramTypes = new LinkedList<>(paramTypesFromDescriptor);

        List<TaskParameter> result = new ArrayList<>();
        while (!paramTypes.isEmpty()) {
            result.add(0, toTaskParameter(paramTypes.pollLast(), taskDetailsBuilder.getStack().pollLast()));
        }
        return result;
    }

    private TaskParameter toTaskParameter(Class<?> paramType, Object param) {
        if (param == null) {
            throw new NullPointerException("You are passing null as a parameter to your background task for type " + paramType.getName() + " - Carrot prevents this to fail fast.");
        }

        if (isClassAssignableToObject(paramType, param)) {
            if (boolean.class.equals(paramType) && Integer.class.equals(param.getClass()))
                return new TaskParameter(paramType, ((Integer) param) > 0);
            return new TaskParameter(paramType, param);
        } else {
            throw shouldNotHappenException(new IllegalStateException("The found parameter types do not match the parameters."));
        }
    }
}
