package cn.boboweike.carrot.tasks.filters;

import cn.boboweike.carrot.CarrotException;
import cn.boboweike.carrot.scheduling.exceptions.TaskNotFoundException;
import cn.boboweike.carrot.tasks.AbstractTask;
import cn.boboweike.carrot.tasks.annotations.Task;
import cn.boboweike.carrot.utils.TaskUtils;
import cn.boboweike.carrot.utils.reflection.ReflectionUtils;
import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Stream;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

public abstract class AbstractTaskFilters {

    protected final AbstractTask task;
    protected final List<TaskFilter> taskFilters;

    protected AbstractTaskFilters(AbstractTask task, TaskDefaultFilters taskDefaultFilters) {
        this.task = task;
        this.taskFilters = initTaskFilters(task, taskDefaultFilters.getFilters());
    }

    protected List<TaskFilter> initTaskFilters(AbstractTask task, List<TaskFilter> taskFilters) {
        try {
            final ArrayList<TaskFilter> result = new ArrayList<>(taskFilters);
            keepOnlyLastElectStateFilter(result);
            addTaskFiltersFromTaskAnnotation(task, result);
            return result;
        } catch (TaskNotFoundException e) {
            return emptyList();
        }
    }

    abstract Logger getLogger();

    private static void keepOnlyLastElectStateFilter(List<TaskFilter> result) {
        if (hasMultipleElectStateFilters(result)) {
            ElectStateFilter firstElectStateFilter = findFirstElectStateFilter(result);
            result.remove(firstElectStateFilter);
        }
    }

    private static void addTaskFiltersFromTaskAnnotation(AbstractTask task, List<TaskFilter> result) {
        final Optional<Task> taskAnnotation = TaskUtils.getTaskAnnotation(task.getTaskDetails());
        if (taskAnnotation.isPresent()) {
            final Optional<ElectStateFilter> electStateFilter = getElectStateFilter(taskAnnotation.get());
            if (electStateFilter.isPresent()) { // only one elect state filter can be present
                result.removeIf(taskFilter -> ElectStateFilter.class.isAssignableFrom(taskFilter.getClass()));
                result.add(electStateFilter.get());
            }
            result.addAll(getOtherTaskFilter(taskAnnotation.get()));
        }
    }

    private static boolean hasMultipleElectStateFilters(List<TaskFilter> result) {
        return result.stream()
                .filter(taskFilter -> ElectStateFilter.class.isAssignableFrom(taskFilter.getClass()))
                .count() > 1;
    }

    private static ElectStateFilter findFirstElectStateFilter(List<TaskFilter> result) {
        return result.stream()
                .filter(taskFilter -> ElectStateFilter.class.isAssignableFrom(taskFilter.getClass()))
                .map(ElectStateFilter.class::cast)
                .findFirst()
                .orElseThrow(() -> CarrotException.shouldNotHappenException("Can not happen..."));
    }

    private static Optional<ElectStateFilter> getElectStateFilter(cn.boboweike.carrot.tasks.annotations.Task taskAnnotation) {
        return Stream.of(taskAnnotation.taskFilters())
                .filter(ElectStateFilter.class::isAssignableFrom)
                .findFirst()
                .map(ReflectionUtils::newInstance)
                .map(ElectStateFilter.class::cast);
    }

    private static List<TaskFilter> getOtherTaskFilter(cn.boboweike.carrot.tasks.annotations.Task taskAnnotation) {
        return Stream.of(taskAnnotation.taskFilters())
                .filter(taskFilter -> !ElectStateFilter.class.isAssignableFrom(taskFilter))
                .map(ReflectionUtils::newInstance)
                .collect(toList());
    }

    <T extends TaskFilter> Consumer<T> catchThrowable(Consumer<T> consumer) {
        return taskClientFilter -> {
            try {
                consumer.accept(taskClientFilter);
            } catch (Exception e) {
                getLogger().error("Error evaluating taskfilter {}", taskClientFilter.getClass().getName(), e);
            }
        };
    }
}