package cn.boboweike.carrot.tasks.filters;

import cn.boboweike.carrot.tasks.AbstractTask;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.annotations.Task;
import cn.boboweike.carrot.utils.StringUtils;
import cn.boboweike.carrot.utils.TaskUtils;
import org.slf4j.MDC;

import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DisplayNameFilter implements TaskClientFilter {

    private static final Pattern regexPattern = Pattern.compile("%X\\{(.*)\\}");

    @Override
    public void onCreating(AbstractTask task) {
        TaskDetails taskDetails = task.getTaskDetails();
        Optional<String> taskNameFromAnnotation = getTaskNameFromAnnotation(taskDetails);
        if (taskNameFromAnnotation.isPresent()) {
            task.setTaskName(getNameWithResolvedParameters(taskNameFromAnnotation.get(), taskDetails));
        } else {
            task.setTaskName(TaskUtils.getReadableNameFromTaskDetails(taskDetails));
        }
    }

    private Optional<String> getTaskNameFromAnnotation(TaskDetails taskDetails) {
        Optional<Task> taskAnnotation = TaskUtils.getTaskAnnotation(taskDetails);
        return taskAnnotation
                .map(Task::name)
                .filter(StringUtils::isNotNullOrEmpty);
    }

    private String getNameWithResolvedParameters(String name, TaskDetails taskDetails) {
        String taskName = replaceTaskParametersInDisplayName(name, taskDetails);
        String result = replaceMDCVariablesInDisplayName(taskName);
        return result;
    }

    private String replaceTaskParametersInDisplayName(String name, TaskDetails taskDetails) {
        String finalName = name;
        for (int i = 0; i < taskDetails.getTaskParameters().size(); i++) {
            finalName = finalName.replace("%" + i, taskDetails.getTaskParameterValues()[i].toString());
        }
        return finalName;
    }

    private String replaceMDCVariablesInDisplayName(String name) {
        Matcher matcher = regexPattern.matcher(name);
        StringBuffer result = new StringBuffer();
        while (matcher.find()) {
            String group0 = matcher.group(0);
            String group1 = matcher.group(1);
            String replacement = MDC.get(matcher.group(1));
            matcher.appendReplacement(result, replacement != null ? replacement : "(" + matcher.group(1) + " is not found in MDC)");
        }
        matcher.appendTail(result);
        return result.toString();
    }
}
