package cn.boboweike.carrot.tasks.filters;

import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.states.FailedState;
import cn.boboweike.carrot.tasks.states.TaskState;
import cn.boboweike.carrot.utils.TaskUtils;

import static cn.boboweike.carrot.tasks.states.StateName.FAILED_STATES;
import static java.time.Instant.now;

/**
 * A TaskFilter of type {@link ElectStateFilter} that will retry the task if it fails for up to 10 times with an exponential back-off policy.
 * This TaskFilter is added by default in Carrot.
 * <p>
 * If you want to configure the amount of retries, create a new instance and pass it to the CarrotConfiguration, e.g.:
 *
 * <pre>
 *     Carrot.configure()
 *                 ...
 *                 .withTaskFilter(new RetryFilter(20, 4)) // this will result in 20 retries and the retries will happen after 4 seconds, 16 seconds, 64 seconds, ...
 *                 ...
 *                 .initialize();
 * </pre>
 */
public class RetryFilter implements ElectStateFilter {

    public static final int DEFAULT_BACKOFF_POLICY_TIME_SEED = 3;
    public static final int DEFAULT_NBR_OF_RETRIES = 10;

    private final int numberOfRetries;
    private final int backOffPolicyTimeSeed;

    public RetryFilter() {
        this(DEFAULT_NBR_OF_RETRIES);
    }

    public RetryFilter(int numberOfRetries) {
        this(numberOfRetries, DEFAULT_BACKOFF_POLICY_TIME_SEED);
    }

    public RetryFilter(int numberOfRetries, int backOffPolicyTimeSeed) {
        this.numberOfRetries = numberOfRetries;
        this.backOffPolicyTimeSeed = backOffPolicyTimeSeed;
    }

    @Override
    public void onStateElection(Task task, TaskState newState) {
        if (isNotFailed(newState) || isProblematicExceptionAndMustNotRetry(newState) || maxAmountOfRetriesReached(task))
            return;

        task.scheduleAt(now().plusSeconds(getSecondsToAdd(task)), String.format("Retry %d of %d", getFailureCount(task), getMaxNumberOfRetries(task)));
    }

    protected long getSecondsToAdd(Task task) {
        return getExponentialBackoffPolicy(task, backOffPolicyTimeSeed);
    }

    protected long getExponentialBackoffPolicy(Task task, int seed) {
        return (long) Math.pow(seed, getFailureCount(task));
    }

    private boolean maxAmountOfRetriesReached(Task task) {
        return getFailureCount(task) > getMaxNumberOfRetries(task);
    }

    private long getFailureCount(Task task) {
        return task.getTaskStates().stream().filter(FAILED_STATES).count();
    }

    private boolean isProblematicExceptionAndMustNotRetry(TaskState newState) {
        if (newState instanceof FailedState) {
            return ((FailedState) newState).mustNotRetry();
        }
        return false;
    }

    private boolean isNotFailed(TaskState newState) {
        return !(newState instanceof FailedState);
    }

    private int getMaxNumberOfRetries(Task task) {
        return TaskUtils.getTaskAnnotation(task.getTaskDetails())
                .map(taskAnnotation -> taskAnnotation.retries() > cn.boboweike.carrot.tasks.annotations.Task.NBR_OF_RETRIES_NOT_PROVIDED ? taskAnnotation.retries() : null)
                .orElse(this.numberOfRetries);
    }
}

