package cn.boboweike.carrot.utils.mapper;

import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.states.AbstractTaskState;
import cn.boboweike.carrot.utils.reflection.ReflectionUtils;
import cn.boboweike.carrot.tasks.TaskParameter;
import cn.boboweike.carrot.tasks.states.EnqueuedState;
import cn.boboweike.carrot.tasks.states.ProcessingState;
import cn.boboweike.carrot.tasks.states.ScheduledState;

import java.io.File;
import java.time.Instant;
import java.util.Arrays;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.setFieldUsingAutoboxing;
import static java.util.Collections.singletonList;

public class JsonMapperValidator {

    public static JsonMapper validateJsonMapper(JsonMapper jsonMapper) {
        try {
            final String serializedTask = jsonMapper.serialize(getTaskForTesting());
            testTimeFields(serializedTask);
            testUseFieldsNotMethods(serializedTask);
            testUsePolymorphism(serializedTask);
            testCanConvertBackToTask(jsonMapper, serializedTask);
            return jsonMapper;
        } catch (Exception e) {
            throw new IllegalArgumentException("The JsonMapper you provided cannot be used as it deserializes tasks in an incorrect way.", e);
        }
    }

    private static void testTimeFields(String serializedTask) {
        if (!serializedTask.contains("\"createdAt\":\"2021-10-14T22:00:00Z\""))
            throw new IllegalArgumentException("Timestamps are wrongly formatted for Carrot. They should be in ISO8601 format.");
    }

    private static void testUseFieldsNotMethods(String serializedTask) {
        if (serializedTask.contains("taskStates") && !serializedTask.contains("taskHistory"))
            throw new IllegalArgumentException("Task Serialization should use fields and not getters/setters.");
    }

    private static void testUsePolymorphism(String serializedTask) {
        if (!serializedTask.contains("\"@class\":\"cn.boboweike.carrot.tasks.states.ScheduledState\""))
            throw new IllegalArgumentException("Polymorphism is not supported as no @class annotation is present with fully qualified name of the different Task states.");
    }

    private static void testCanConvertBackToTask(JsonMapper jsonMapper, String serializedTask) {
        try {
            jsonMapper.deserialize(serializedTask, Task.class);
        } catch (Exception e) {
            throw new IllegalArgumentException("The JsonMapper cannot convert tasks from Json.", e);
        }
    }

    private static <T extends AbstractTaskState> T setCreatedAndScheduledDates(T taskState, Instant createdAt, Instant scheduledAt) {
        setCreatedAt(taskState, createdAt);
        setField(taskState, "scheduledAt", scheduledAt);
        return taskState;
    }

    private static <T extends AbstractTaskState> T setCreatedAt(T taskState, Instant instant) {
        setField(taskState, "createdAt", instant);
        return taskState;
    }

    private static <T extends AbstractTaskState> T setField(T taskState, String fieldname, Instant instant) {
        ReflectionUtils.setFieldUsingAutoboxing(fieldname, taskState, instant);
        return taskState;
    }

    private static Task getTaskForTesting() {
        final Task task = new Task(
                UUID.randomUUID(),
                5,
                new TaskDetails("java.lang.System", "out", "println", singletonList(
                        new TaskParameter("java.io.File", new File("/tmp/"))
                )),

                Arrays.asList(
                        setCreatedAndScheduledDates(new ScheduledState(Instant.now()), Instant.ofEpochSecond(1634248800), Instant.ofEpochSecond(1634245200)),
                        setCreatedAt(new EnqueuedState(), Instant.ofEpochSecond(1634248800)),
                        setCreatedAt(new ProcessingState(UUID.fromString("117bbfcf-e6df-45f0-82a7-b88fd8f96c06")), Instant.ofEpochSecond(1634248900))),
                new ConcurrentHashMap<>()
        );
        task.setTaskName("Some name");
        return task;
    }
}
