package cn.boboweike.carrot.utils.mapper.gson;

import cn.boboweike.carrot.CarrotException;
import cn.boboweike.carrot.tasks.TaskParameter;
import cn.boboweike.carrot.tasks.context.TaskContext;
import cn.boboweike.carrot.tasks.states.TaskState;
import cn.boboweike.carrot.utils.mapper.JsonMapper;
import cn.boboweike.carrot.utils.mapper.TaskParameterJsonMapperException;
import cn.boboweike.carrot.utils.metadata.VersionRetriever;
import cn.boboweike.carrot.utils.reflection.ReflectionUtils;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapterFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.reflect.Field;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static java.util.Collections.unmodifiableList;

public class GsonJsonMapper implements JsonMapper {

    private final Gson gson;

    public GsonJsonMapper() {
        this(new GsonBuilder());
    }

    public GsonJsonMapper(GsonBuilder gsonBuilder) {
        this.gson = initGson(gsonBuilder);
        fixGsonNotBeingExtensible(gson);
    }

    public GsonJsonMapper(Gson gson) {
        this.gson = gson;
        fixGsonNotBeingExtensible(gson);
    }

    protected Gson initGson(GsonBuilder gsonBuilder) {
        return gsonBuilder
                .serializeNulls()
                .registerTypeAdapterFactory(RuntimeClassNameTypeAdapterFactory.of(TaskState.class))
                .registerTypeAdapterFactory(RuntimeClassNameTypeAdapterFactory.of(Map.class))
                .registerTypeAdapterFactory(RuntimeClassNameTypeAdapterFactory.of(TaskContext.Metadata.class))
                .registerTypeHierarchyAdapter(Path.class, new PathAdapter().nullSafe())
                .registerTypeAdapter(Instant.class, new InstantAdapter().nullSafe())
                .registerTypeAdapter(Duration.class, new DurationAdapter())
                .registerTypeAdapter(TaskParameter.class, new TaskParameterDeserializer())
                .create();
    }

    @Override
    public String serialize(Object object) {
        try {
            return gson.toJson(object);
        } catch (Exception e) {
            throw new TaskParameterJsonMapperException("The task parameters are not serializable.", e);
        }
    }

    @Override
    public void serialize(OutputStream outputStream, Object object) {
        try (final OutputStreamWriter writer = new OutputStreamWriter(outputStream)) {
            gson.toJson(object, writer);
        } catch (IOException e) {
            throw CarrotException.shouldNotHappenException(e);
        }
    }

    @Override
    public <T> T deserialize(String serializedObjectAsString, Class<T> clazz) {
        return gson.fromJson(serializedObjectAsString, clazz);
    }

    // I'm really sorry for this
    // see https://github.com/google/gson/issues/1177
    // when deserializing, GSON does not create correct types for items in collections
    private void fixGsonNotBeingExtensible(Gson gson) {
        try {
            final Field factories = ReflectionUtils.getField(Gson.class, "factories");
            ReflectionUtils.makeAccessible(factories);
            final List o = new ArrayList<TypeAdapterFactory>((Collection<? extends TypeAdapterFactory>) factories.get(gson));
            if (!o.get(1).getClass().getName().contains("ObjectTypeAdapter"))
                throw CarrotException.shouldNotHappenException(String.format("It looks like you are running a Gson version (%s) which is not compatible with Carrot", VersionRetriever.getVersion(Gson.class)));
            o.set(1, ClassNameObjectTypeAdapter.FACTORY);
            factories.set(gson, unmodifiableList(o));
        } catch (ReflectiveOperationException e) {
            throw CarrotException.shouldNotHappenException(e);
        }
    }
}