package cn.boboweike.carrot.utils.mapper.jackson.modules;

import cn.boboweike.carrot.tasks.TaskParameterNotDeserializableException;
import cn.boboweike.carrot.utils.mapper.JsonMapperUtils;
import cn.boboweike.carrot.utils.reflection.ReflectionUtils;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.exc.MismatchedInputException;
import com.fasterxml.jackson.databind.node.ArrayNode;
import cn.boboweike.carrot.tasks.TaskParameter;

import java.util.Collection;

public class TaskParameterDeserializer extends StdDeserializer<TaskParameter> {

    private final ObjectMapper objectMapper;

    protected TaskParameterDeserializer() {
        super(TaskParameter.class);
        this.objectMapper = new ObjectMapper();
    }

    @Override
    public TaskParameter deserialize(JsonParser jsonParser, DeserializationContext deserializationContext) throws IOException {
        JsonNode node = jsonParser.getCodec().readTree(jsonParser);
        final String className = node.get(JsonMapperUtils.Json.FIELD_CLASS_NAME).asText();
        final String actualClassName = node.has(JsonMapperUtils.Json.FIELD_ACTUAL_CLASS_NAME) ? node.get(JsonMapperUtils.Json.FIELD_ACTUAL_CLASS_NAME).asText() : null;
        final JsonNode objectJsonNode = node.get("object");
        if (Path.class.getName().equals(className)) { // see https://github.com/FasterXML/jackson-databind/issues/2013
            return new TaskParameter(className, Paths.get(objectJsonNode.asText().replace("file:", "")));
        } else {
            return getTaskParameter(jsonParser, className, actualClassName, objectJsonNode);
        }
    }

    private TaskParameter getTaskParameter(JsonParser jsonParser, String className, String actualClassName, JsonNode objectJsonNode) throws JsonProcessingException {
        try {
            Class<Object> valueType = ReflectionUtils.toClass(getActualClassName(className, actualClassName));
            if (objectJsonNode.isArray() && !Collection.class.isAssignableFrom(valueType)) { // why: regression form 4.0.1: See https://github.com/boboweike/carrot/issues/254
                final JsonNode jsonNodeInArray = objectJsonNode.get(1);
                final Object object = jsonParser.getCodec().treeToValue(jsonNodeInArray, valueType);
                return new TaskParameter(className, object);
            } else {
                try {
                    final Object object = jsonParser.getCodec().treeToValue(objectJsonNode, valueType);
                    return new TaskParameter(className, object);
                } catch (MismatchedInputException e) { // last attempts
                    // is it an Enum?
                    if (valueType.isEnum()) {
                        ArrayNode arrayNode = (ArrayNode) jsonParser.getCodec().createArrayNode();
                        arrayNode.add(valueType.getName());
                        arrayNode.add(objectJsonNode);
                        final Object object = jsonParser.getCodec().treeToValue(arrayNode, valueType);
                        return new TaskParameter(className, object);
                    } else {
                        // another try for Kotlin
                        final Object object = objectMapper.treeToValue(objectJsonNode, valueType);
                        return new TaskParameter(className, object);
                    }
                }
            }
        } catch (Exception e) {
            return new TaskParameter(new TaskParameterNotDeserializableException(getActualClassName(className, actualClassName), e.getMessage()));
        }
    }

    public static String getActualClassName(String methodClassName, String actualClassName) {
        return JsonMapperUtils.getActualClassName(methodClassName, actualClassName, "sun.", "com.sun.");
    }
}

