/*
 * Decompiled with CFR 0.152.
 */
package cn.boboweike.carrot.scheduling;

import cn.boboweike.carrot.configuration.Carrot;
import cn.boboweike.carrot.fixtures.CarrotAssertions;
import cn.boboweike.carrot.fixtures.storage.PartitionedStorageProviderForTest;
import cn.boboweike.carrot.fixtures.stubs.StaticTestService;
import cn.boboweike.carrot.fixtures.stubs.TestService;
import cn.boboweike.carrot.fixtures.tasks.TaskDetailsTestBuilder;
import cn.boboweike.carrot.fixtures.tasks.TaskTestBuilder;
import cn.boboweike.carrot.scheduling.BackgroundTask;
import cn.boboweike.carrot.scheduling.cron.Cron;
import cn.boboweike.carrot.scheduling.exceptions.TaskClassNotFoundException;
import cn.boboweike.carrot.scheduling.exceptions.TaskMethodNotFoundException;
import cn.boboweike.carrot.server.BackgroundTaskServer;
import cn.boboweike.carrot.server.BackgroundTaskServerConfiguration;
import cn.boboweike.carrot.storage.InMemoryPartitionedStorageProvider;
import cn.boboweike.carrot.storage.PageRequest;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.tasks.context.TaskContext;
import cn.boboweike.carrot.tasks.lambdas.TaskLambda;
import cn.boboweike.carrot.tasks.lambdas.TaskLambdaFromStream;
import cn.boboweike.carrot.tasks.states.FailedState;
import cn.boboweike.carrot.tasks.states.ProcessingState;
import cn.boboweike.carrot.tasks.states.StateName;
import cn.boboweike.carrot.tasks.states.TaskState;
import io.github.artsok.RepeatedIfExceptionsTest;
import java.io.Serializable;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.awaitility.Awaitility;
import org.awaitility.Durations;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.MDC;

public class BackgroundTaskByTaskLambdaTest {
    private TestService testService;
    private PartitionedStorageProviderForTest storageProvider;
    private BackgroundTaskServer backgroundTaskServer;
    private static final String every5Seconds = "*/5 * * * * *";

    @BeforeEach
    void setUpTests() {
        this.testService = new TestService();
        this.testService.reset();
        this.storageProvider = new PartitionedStorageProviderForTest((PartitionedStorageProvider)new InMemoryPartitionedStorageProvider());
        Carrot.configure().useStorageProvider((PartitionedStorageProvider)this.storageProvider).useBackgroundTaskServer(BackgroundTaskServerConfiguration.usingStandardBackgroundTaskServerConfiguration().andPollIntervalInSeconds(5)).initialize();
        this.backgroundTaskServer = Carrot.getBackgroundTaskServer();
    }

    @AfterEach
    void cleanUp() {
        MDC.clear();
        this.backgroundTaskServer.stop();
    }

    @Test
    void ifBackgroundTaskIsNotConfiguredCorrectlyAnExceptionIsThrown() {
        BackgroundTask.setTaskScheduler(null);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> BackgroundTask.enqueue((TaskLambda & Serializable)() -> System.out.println("Test"))).isInstanceOf(IllegalStateException.class)).hasMessage("The TaskScheduler has not been initialized. Use the fluent Carrot.configure() API to setup Carrot or set the TaskScheduler via the static setter method.");
    }

    @Test
    void testEnqueueSystemOut() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> System.out.println("this is a test"));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testEnqueue() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWork());
        Awaitility.await().atMost(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testEnqueueWithId() {
        UUID id = UUID.randomUUID();
        TaskId taskId1 = BackgroundTask.enqueue((UUID)id, (TaskLambda & Serializable)() -> this.testService.doWork());
        TaskId taskId2 = BackgroundTask.enqueue((UUID)id, (TaskLambda & Serializable)() -> this.testService.doWork());
        CarrotAssertions.assertThat((Object)taskId1).isEqualTo((Object)taskId2);
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId1)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED));
    }

    @Test
    public void testEnqueueWithStaticMethod() {
        TaskId taskId = BackgroundTask.enqueue(TestService::doStaticWork);
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED));
    }

    @Test
    public void testEnqueueWithStaticMethodWithArgument() {
        UUID id = UUID.randomUUID();
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> StaticTestService.doWorkInStaticMethod(id));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED));
    }

    @Test
    void testEnqueueWithInterfaceImplementationThrowsNiceException() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> BackgroundTask.enqueue((TaskLambda)new TaskImplementation())).isInstanceOf(IllegalArgumentException.class)).hasMessage("Please provide a lambda expression (e.g. BackgroundTask.enqueue(() -> myService.doWork()) instead of an actual implementation.");
    }

    @Test
    void testEnqueueWithCustomObject() {
        TestService.Work work = new TestService.Work(2, "some string", UUID.randomUUID());
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWork(work));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testEnqueueWithPath() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkWithPath(Paths.get("/tmp/carrot/example.log", new String[0])));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testEnqueueWithTaskContextAndMetadata() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWork((Integer)5, TaskContext.Null));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.PROCESSING);
        Awaitility.await().atMost(Durations.TEN_SECONDS).until(() -> !this.storageProvider.getTaskById(taskId).getMetadata().isEmpty());
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasMetadata("test", "test");
        Awaitility.await().atMost(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED).hasMetadataOnlyContainingTaskProgressAndLogging();
    }

    @Test
    void testEnqueueStreamWithMultipleParameters() {
        Stream<UUID> workStream = this.getWorkStream();
        AtomicInteger atomicInteger = new AtomicInteger();
        BackgroundTask.enqueue(workStream, (TaskLambdaFromStream & Serializable)uuid -> this.testService.doWork(uuid.toString(), atomicInteger.incrementAndGet(), Instant.now()));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat((Long)this.storageProvider.countTasks(StateName.SUCCEEDED)).isEqualTo(5L));
    }

    @Test
    void testEnqueueStreamWithWrappingObjectAsParameter() {
        AtomicInteger atomicInteger = new AtomicInteger();
        Stream<TestService.Work> workStream = this.getWorkStream().map(uuid -> new TestService.Work(atomicInteger.incrementAndGet(), "some string " + uuid, (UUID)uuid));
        BackgroundTask.enqueue(workStream, (TaskLambdaFromStream & Serializable)work -> this.testService.doWork((TestService.Work)work));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat((Long)this.storageProvider.countTasks(StateName.SUCCEEDED)).isEqualTo(5L));
    }

    @Test
    void testEnqueueStreamWithParameterFromWrappingObject() {
        AtomicInteger atomicInteger = new AtomicInteger();
        Stream<TestService.Work> workStream = this.getWorkStream().map(uuid -> new TestService.Work(atomicInteger.incrementAndGet(), "some string " + uuid, (UUID)uuid));
        BackgroundTask.enqueue(workStream, (TaskLambdaFromStream & Serializable)work -> this.testService.doWork(work.getUuid()));
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat((Long)this.storageProvider.countTasks(StateName.SUCCEEDED)).isEqualTo(5L));
    }

    @Test
    void testEnqueueStreamWithMethodReference() {
        BackgroundTask.enqueue(this.getWorkStream(), TestService::doWorkWithUUID);
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat((Long)this.storageProvider.countTasks(StateName.SUCCEEDED)).isEqualTo(5L));
    }

    @Test
    void tasksCanEnqueueOtherTasksInTheSameClassUsingMethodReference() {
        TaskId taskId = BackgroundTask.enqueue(this::aNestedTask);
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED));
    }

    @Test
    void testFailedTaskAddsFailedStateAndScheduledThanksToDefaultRetryFilter() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkThatFails());
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.FAILED, StateName.SCHEDULED));
    }

    @Test
    void testScheduleWithId() {
        UUID id = UUID.randomUUID();
        TaskId taskId1 = BackgroundTask.schedule((UUID)id, (Instant)Instant.now(), (TaskLambda & Serializable)() -> this.testService.doWork());
        TaskId taskId2 = BackgroundTask.schedule((UUID)id, (Instant)Instant.now().plusSeconds(20L), (TaskLambda & Serializable)() -> this.testService.doWork());
        CarrotAssertions.assertThat((Object)taskId1).isEqualTo((Object)taskId2);
        Awaitility.await().atMost(Durations.FIVE_SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId1)).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED));
    }

    @Test
    void testScheduleWithZonedDateTime() {
        TaskId taskId = BackgroundTask.schedule((ZonedDateTime)ZonedDateTime.now().plusSeconds(7L), (TaskLambda & Serializable)() -> this.testService.doWork());
        Awaitility.await().during(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SCHEDULED);
        Awaitility.await().atMost(Durations.TEN_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testScheduleWithOffsetDateTime() {
        TaskId taskId = BackgroundTask.schedule((OffsetDateTime)OffsetDateTime.now().plusSeconds(7L), (TaskLambda & Serializable)() -> this.testService.doWork());
        Awaitility.await().during(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SCHEDULED);
        Awaitility.await().atMost(Durations.TEN_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testScheduleWithLocalDateTime() {
        TaskId taskId = BackgroundTask.schedule((LocalDateTime)LocalDateTime.now().plusSeconds(7L), (TaskLambda & Serializable)() -> this.testService.doWork());
        Awaitility.await().during(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SCHEDULED);
        Awaitility.await().atMost(Durations.TEN_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testScheduleWithInstant() {
        TaskId taskId = BackgroundTask.schedule((Instant)Instant.now().plusSeconds(7L), (TaskLambda & Serializable)() -> this.testService.doWork());
        Awaitility.await().during(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SCHEDULED);
        Awaitility.await().atMost(Durations.TEN_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testScheduleUsingDateTimeInTheFutureIsNotEnqueued() {
        TaskId taskId = BackgroundTask.schedule((Instant)Instant.now().plus(100L, ChronoUnit.DAYS), (TaskLambda & Serializable)() -> this.testService.doWork());
        Awaitility.await().during(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SCHEDULED);
        Awaitility.await().atMost(Durations.FIVE_SECONDS).until(() -> this.storageProvider.getTaskById(taskId).getState() == StateName.SCHEDULED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED);
    }

    @Test
    void testScheduleThatSchedulesOtherTasks() {
        TaskId taskId = BackgroundTask.schedule((Instant)Instant.now().plusSeconds(1L), (TaskLambda & Serializable)() -> this.testService.scheduleNewWork(5));
        Awaitility.await().atMost(Durations.ONE_MINUTE).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 6L);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testScheduleThatSchedulesOtherTasksSlowlyDoesNotBlockOtherWorkers() {
        TaskId taskId = BackgroundTask.schedule((Instant)Instant.now().plusSeconds(1L), (TaskLambda & Serializable)() -> this.testService.scheduleNewWorkSlowly(5));
        Awaitility.await().atMost(Duration.ofSeconds(12L)).until(() -> this.storageProvider.countTasks(StateName.PROCESSING) + this.storageProvider.countTasks(StateName.SUCCEEDED) > 1L);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING);
    }

    @Test
    void testRecurringCronTask() {
        BackgroundTask.scheduleRecurrently((String)every5Seconds, (TaskLambda & Serializable)() -> this.testService.doWork(5));
        Awaitility.await().atMost(65L, TimeUnit.SECONDS).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 1L);
        Task task = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(0);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testRecurringCronTaskWithTaskContext() {
        BackgroundTask.scheduleRecurrently((String)every5Seconds, (TaskLambda & Serializable)() -> this.testService.doWork((Integer)5, TaskContext.Null));
        Awaitility.await().atMost(65L, TimeUnit.SECONDS).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 1L);
        Task task = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(0);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testRecurringCronTaskWithId() {
        BackgroundTask.scheduleRecurrently((String)"theId", (String)every5Seconds, (TaskLambda & Serializable)() -> this.testService.doWork(5));
        Awaitility.await().atMost(25L, TimeUnit.SECONDS).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 1L);
        Task task = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(0);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testRecurringCronTaskWithIdAndTimezone() {
        BackgroundTask.scheduleRecurrently((String)"theId", (String)every5Seconds, (ZoneId)ZoneId.systemDefault(), (TaskLambda & Serializable)() -> this.testService.doWork(5));
        Awaitility.await().atMost(25L, TimeUnit.SECONDS).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 1L);
        Task task = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(0);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testRecurringIntervalTask() {
        BackgroundTask.scheduleRecurrently((Duration)Duration.ofSeconds(5L), (TaskLambda & Serializable)() -> this.testService.doWork(5));
        Awaitility.await().atMost(15L, TimeUnit.SECONDS).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 1L);
        Task task = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(0);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testRecurringIntervalTaskWithId() {
        BackgroundTask.scheduleRecurrently((String)"theId", (Duration)Duration.ofSeconds(5L), (TaskLambda & Serializable)() -> this.testService.doWork(5));
        Awaitility.await().atMost(15L, TimeUnit.SECONDS).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 1L);
        Task task = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(0);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void test2RecurringTasksWithSameMethodSignatureShouldBothBeRun() {
        BackgroundTask.scheduleRecurrently((String)"recurring-task-1", (String)every5Seconds, (ZoneId)ZoneId.systemDefault(), (TaskLambda & Serializable)() -> this.testService.doWork(5));
        BackgroundTask.scheduleRecurrently((String)"recurring-task-2", (String)every5Seconds, (ZoneId)ZoneId.systemDefault(), (TaskLambda & Serializable)() -> this.testService.doWork(5));
        Awaitility.await().atMost(25L, TimeUnit.SECONDS).until(() -> this.storageProvider.countTasks(StateName.SUCCEEDED) == 2L);
        Task task1 = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(0);
        Task task2 = this.storageProvider.getTasksByPartition(StateName.SUCCEEDED, PageRequest.ascOnUpdatedAt((int)1000), 0).get(1);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task1.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(task2.getId())).hasStates(StateName.SCHEDULED, StateName.ENQUEUED, StateName.PROCESSING, StateName.SUCCEEDED);
    }

    @Test
    void testDeleteOfRecurringTask() {
        String taskId = BackgroundTask.scheduleRecurrently((String)Cron.minutely(), (TaskLambda & Serializable)() -> this.testService.doWork(5));
        BackgroundTask.delete((String)taskId);
        CarrotAssertions.assertThat(this.storageProvider.getRecurringTasks()).isEmpty();
    }

    @Test
    void tasksStuckInProcessingStateAreRescheduled() {
        Task task = this.storageProvider.save(TaskTestBuilder.anEnqueuedTask().withState((TaskState)new ProcessingState(this.backgroundTaskServer.getId()), Instant.now().minus(15L, ChronoUnit.MINUTES)).build());
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(task.getId())).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.FAILED, StateName.SCHEDULED));
    }

    @Test
    void taskCanBeUpdatedInTheBackgroundAndThenGoToSucceededState() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkThatTakesLong(10));
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(taskId).hasState(StateName.PROCESSING));
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> {
            Task task = this.storageProvider.getTaskById(taskId);
            ProcessingState processingState = (ProcessingState)task.getTaskState();
            CarrotAssertions.assertThat((Instant)processingState.getUpdatedAt()).isAfter(processingState.getCreatedAt());
            this.storageProvider.getTaskById(taskId).hasState(StateName.PROCESSING);
        });
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasState(StateName.SUCCEEDED));
    }

    @RepeatedIfExceptionsTest(repeats=3)
    void taskCanBeDeletedWhenEnqueued() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkThatTakesLong(12));
        BackgroundTask.delete((TaskId)taskId);
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> {
            CarrotAssertions.assertThat((int)this.backgroundTaskServer.getTaskZooKeeper().getOccupiedWorkerCount()).isZero();
            CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.DELETED);
        });
    }

    @Test
    void taskCanBeDeletedWhenScheduled() {
        TaskId taskId = BackgroundTask.schedule((Instant)Instant.now().plusSeconds(10L), (TaskLambda & Serializable)() -> this.testService.doWorkThatTakesLong(12));
        BackgroundTask.delete((TaskId)taskId);
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> {
            CarrotAssertions.assertThat((int)this.backgroundTaskServer.getTaskZooKeeper().getOccupiedWorkerCount()).isZero();
            CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.SCHEDULED, StateName.DELETED);
        });
    }

    @Test
    void taskCanBeDeletedDuringProcessingState_taskRethrowsInterruptedException() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkThatTakesLong(12));
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(taskId).hasState(StateName.PROCESSING));
        BackgroundTask.delete((TaskId)taskId);
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> {
            CarrotAssertions.assertThat((int)this.backgroundTaskServer.getTaskZooKeeper().getOccupiedWorkerCount()).isZero();
            CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.DELETED);
        });
        Awaitility.await().during(6L, TimeUnit.SECONDS).atMost(12L, TimeUnit.SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).doesNotHaveState(StateName.SUCCEEDED));
    }

    @Test
    void taskCanBeDeletedDuringProcessingState_taskInterruptCurrentThread() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkThatTakesLongInterruptThread(12));
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(taskId).hasState(StateName.PROCESSING));
        BackgroundTask.delete((TaskId)taskId);
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> {
            CarrotAssertions.assertThat((int)this.backgroundTaskServer.getTaskZooKeeper().getOccupiedWorkerCount()).isZero();
            CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.DELETED);
        });
        Awaitility.await().during(12L, TimeUnit.SECONDS).atMost(18L, TimeUnit.SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).doesNotHaveState(StateName.SUCCEEDED));
    }

    @Test
    void taskCanBeDeletedDuringProcessingStateIfInterruptible() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkThatCanBeInterrupted(12));
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(taskId).hasState(StateName.PROCESSING));
        BackgroundTask.delete((TaskId)taskId);
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> {
            CarrotAssertions.assertThat((int)this.backgroundTaskServer.getTaskZooKeeper().getOccupiedWorkerCount()).isZero();
            CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.DELETED);
        });
        Awaitility.await().during(12L, TimeUnit.SECONDS).atMost(18L, TimeUnit.SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).doesNotHaveState(StateName.SUCCEEDED));
    }

    @Test
    void taskCanBeDeletedDuringProcessingState_InterruptedExceptionCatched() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkThatTakesLongCatchInterruptException(12));
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(taskId).hasState(StateName.PROCESSING));
        BackgroundTask.delete((TaskId)taskId);
        Awaitility.await().atMost(6L, TimeUnit.SECONDS).untilAsserted(() -> {
            CarrotAssertions.assertThat((int)this.backgroundTaskServer.getTaskZooKeeper().getOccupiedWorkerCount()).isZero();
            CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.DELETED);
        });
        Awaitility.await().during(12L, TimeUnit.SECONDS).atMost(18L, TimeUnit.SECONDS).untilAsserted(() -> CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).doesNotHaveState(StateName.SUCCEEDED));
    }

    @Test
    void processingCanBeSkippedUsingElectStateFilters() {
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.tryToDoWorkButDontBecauseOfSomeBusinessRuleDefinedInTheOnStateElectionFilter());
        Awaitility.await().during(3L, TimeUnit.SECONDS).atMost(6L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(taskId).hasState(StateName.SCHEDULED));
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.DELETED, StateName.SCHEDULED);
    }

    @Test
    void taskToClassThatDoesNotExistGoesToFailedState() {
        Task task = this.storageProvider.save(TaskTestBuilder.anEnqueuedTask().withTaskDetails(TaskDetailsTestBuilder.classThatDoesNotExistTaskDetails()).build());
        Awaitility.await().atMost(3L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(task.getId()).hasState(StateName.FAILED));
        FailedState failedState = (FailedState)this.storageProvider.getTaskById(task.getId()).getTaskState();
        CarrotAssertions.assertThat((Throwable)failedState.getException()).isInstanceOf(TaskClassNotFoundException.class);
        Task failedTask = this.storageProvider.getTaskById(task.getId());
        CarrotAssertions.assertThat(failedTask).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.FAILED);
    }

    @Test
    void taskToMethodThatDoesNotExistGoesToFailedState() {
        Task task = this.storageProvider.save(TaskTestBuilder.anEnqueuedTask().withTaskDetails(TaskDetailsTestBuilder.methodThatDoesNotExistTaskDetails()).build());
        Awaitility.await().atMost(30L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(task.getId()).hasState(StateName.FAILED));
        FailedState failedState = (FailedState)this.storageProvider.getTaskById(task.getId()).getTaskState();
        CarrotAssertions.assertThat((Throwable)failedState.getException()).isInstanceOf(TaskMethodNotFoundException.class);
        Awaitility.await().during(1L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(task.getId()).hasState(StateName.FAILED));
        Task failedTask = this.storageProvider.getTaskById(task.getId());
        CarrotAssertions.assertThat(failedTask).hasStates(StateName.ENQUEUED, StateName.PROCESSING, StateName.FAILED);
    }

    @Test
    void testTaskInheritance() {
        SomeSysoutTaskClass someSysoutTaskClass = new SomeSysoutTaskClass(Cron.daily());
        Assertions.assertThatCode(() -> someSysoutTaskClass.schedule()).doesNotThrowAnyException();
    }

    @Test
    void mdcContextIsAvailableInTask() {
        MDC.put((String)"someKey", (String)"someValue");
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkWithMDC("someKey"));
        Awaitility.await().atMost(30L, TimeUnit.SECONDS).until(() -> this.storageProvider.getTaskById(taskId).hasState(StateName.SUCCEEDED));
    }

    @Test
    void mdcContextIsAvailableForDisplayName() {
        MDC.put((String)"customer.id", (String)"1");
        TaskId taskId = BackgroundTask.enqueue((TaskLambda & Serializable)() -> this.testService.doWorkWithAnnotation(5, "John Doe"));
        CarrotAssertions.assertThat(this.storageProvider.getTaskById(taskId)).hasTaskName("Doing some hard work for user John Doe (customerId: 1)");
    }

    public void aNestedTask() {
        System.out.println("Nothing else to do");
    }

    private Stream<UUID> getWorkStream() {
        return IntStream.range(0, 5).mapToObj(i -> UUID.randomUUID());
    }

    static class TaskImplementation
    implements TaskLambda {
        TaskImplementation() {
        }

        public void run() throws Exception {
            System.out.println("This should not be run");
        }
    }

    public static class SomeSysoutTaskClass
    extends SomeTaskClass {
        public SomeSysoutTaskClass(String cron) {
            super(cron);
        }

        @Override
        public void doWork() {
            System.out.println("In doWork method");
        }
    }

    static abstract class SomeTaskClass
    implements SomeTaskInterface {
        private final String cron;

        public SomeTaskClass(String cron) {
            this.cron = cron;
        }

        public void schedule() {
            BackgroundTask.scheduleRecurrently((String)"test-id", (String)this.cron, (TaskLambda & Serializable)() -> this.doWork());
        }
    }

    static interface SomeTaskInterface {
        public void doWork();
    }
}

