/*
 * Decompiled with CFR 0.152.
 */
package cn.boboweike.carrot.server;

import cn.boboweike.carrot.scheduling.exceptions.TaskNotFoundException;
import cn.boboweike.carrot.server.BackgroundTaskServer;
import cn.boboweike.carrot.server.runner.BackgroundTaskRunner;
import cn.boboweike.carrot.storage.ConcurrentTaskModificationException;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.context.CarrotDashboardLogger;
import cn.boboweike.carrot.tasks.filters.TaskPerformingFilters;
import cn.boboweike.carrot.tasks.mappers.MDCMapper;
import cn.boboweike.carrot.tasks.states.IllegalTaskStateChangeException;
import cn.boboweike.carrot.tasks.states.StateName;
import cn.boboweike.carrot.utils.annotations.VisibleFor;
import cn.boboweike.carrot.utils.exceptions.Exceptions;
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class BackgroundTaskPerformer
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackgroundTaskPerformer.class);
    private static final AtomicInteger concurrentModificationExceptionCounter = new AtomicInteger();
    private final BackgroundTaskServer backgroundTaskServer;
    private final TaskPerformingFilters taskPerformingFilters;
    private final Task task;

    public BackgroundTaskPerformer(BackgroundTaskServer backgroundTaskServer, Task task) {
        this.backgroundTaskServer = backgroundTaskServer;
        this.taskPerformingFilters = new TaskPerformingFilters(task, backgroundTaskServer.getTaskFilters());
        this.task = task;
    }

    @Override
    public void run() {
        try {
            this.backgroundTaskServer.getTaskZooKeeper().notifyThreadOccupied();
            boolean canProcess = this.updateTaskStateToProcessingRunTaskFiltersAndReturnIfProcessingCanStart();
            if (canProcess) {
                this.runActualTask();
                this.updateTaskStateToSucceededAndRunTaskFilters();
            }
        }
        catch (Exception e) {
            if (this.isTaskDeletedWhileProcessing(e)) {
                return;
            }
            if (this.isTaskServerStopped(e)) {
                this.updateTaskStateToFailedAndRunTaskFilters("Task processing was stopped as background task server has stopped", e);
                Thread.currentThread().interrupt();
            } else if (this.isTaskNotFoundException(e)) {
                this.updateTaskStateToFailedAndRunTaskFilters("Task method not found", e);
            } else {
                this.updateTaskStateToFailedAndRunTaskFilters("An exception occurred during the performance of the task", e);
            }
        }
        finally {
            this.backgroundTaskServer.getTaskZooKeeper().notifyThreadIdle();
        }
    }

    private boolean updateTaskStateToProcessingRunTaskFiltersAndReturnIfProcessingCanStart() {
        try {
            this.task.startProcessingOn(this.backgroundTaskServer);
            this.saveAndRunStateRelatedTaskFilters(this.task);
            LOGGER.debug("Task(id={}, taskName='{}') processing started", (Object)this.task.getId(), (Object)this.task.getTaskName());
            return this.task.hasState(StateName.PROCESSING);
        }
        catch (ConcurrentTaskModificationException e) {
            LOGGER.trace("Could not start processing task {} - it is already in a newer state (collision {})", (Object)this.task.getId(), (Object)concurrentModificationExceptionCounter.incrementAndGet());
            return false;
        }
    }

    private void runActualTask() throws Exception {
        try {
            MDCMapper.loadMDCContextFromTask(this.task);
            CarrotDashboardLogger.setTask(this.task);
            this.backgroundTaskServer.getTaskZooKeeper().startProcessing(this.task, Thread.currentThread());
            LOGGER.trace("Task(id={}, taskName='{}') is running", (Object)this.task.getId(), (Object)this.task.getTaskName());
            this.taskPerformingFilters.runOnTaskProcessingFilters();
            BackgroundTaskRunner backgroundTaskRunner = this.backgroundTaskServer.getBackgroundTaskRunner(this.task);
            backgroundTaskRunner.run(this.task);
            this.taskPerformingFilters.runOnTaskProcessedFilters();
        }
        finally {
            this.backgroundTaskServer.getTaskZooKeeper().stopProcessing(this.task);
            CarrotDashboardLogger.clearTask();
            MDC.clear();
        }
    }

    private void updateTaskStateToSucceededAndRunTaskFilters() {
        try {
            LOGGER.debug("Task(id={}, taskName='{}') processing succeeded", (Object)this.task.getId(), (Object)this.task.getTaskName());
            this.task.succeeded();
            this.saveAndRunStateRelatedTaskFilters(this.task);
        }
        catch (IllegalTaskStateChangeException ex) {
            if (ex.getFrom() == StateName.DELETED) {
                LOGGER.info("Task finished successfully but it was already deleted - ignoring illegal state change from {} to {}", (Object)ex.getFrom(), (Object)ex.getTo());
            }
            throw ex;
        }
        catch (Exception badException) {
            LOGGER.error("ERROR - could not update task(id={}, taskName='{}') to SUCCEEDED state", new Object[]{this.task.getId(), this.task.getTaskName(), badException});
        }
    }

    private void updateTaskStateToFailedAndRunTaskFilters(String message, Exception e) {
        try {
            Exception actualException = BackgroundTaskPerformer.unwrapException(e);
            this.task.failed(message, actualException);
            this.saveAndRunStateRelatedTaskFilters(this.task);
            if (this.task.getState() == StateName.FAILED) {
                LOGGER.error("Task(id={}, taskName='{}') processing failed: {}", new Object[]{this.task.getId(), this.task.getTaskName(), message, actualException});
            } else {
                LOGGER.warn("Task(id={}, taskName='{}') processing failed: {}", new Object[]{this.task.getId(), this.task.getTaskName(), message, actualException});
            }
        }
        catch (IllegalTaskStateChangeException ex) {
            if (ex.getFrom() == StateName.DELETED) {
                LOGGER.info("Task processing failed but it was already deleted - ignoring illegal state change from {} to {}", (Object)ex.getFrom(), (Object)ex.getTo());
            }
            throw ex;
        }
        catch (Exception badException) {
            LOGGER.error("ERROR - could not update task(id={}, taskName='{}') to FAILED state", new Object[]{this.task.getId(), this.task.getTaskName(), badException});
        }
    }

    protected void saveAndRunStateRelatedTaskFilters(Task task) {
        this.taskPerformingFilters.runOnStateAppliedFilters();
        StateName beforeStateElection = task.getState();
        this.taskPerformingFilters.runOnStateElectionFilter();
        StateName afterStateElection = task.getState();
        this.backgroundTaskServer.getStorageProvider().saveByPartition(task, this.getPartition());
        if (beforeStateElection != afterStateElection) {
            this.taskPerformingFilters.runOnStateAppliedFilters();
        }
    }

    private Integer getPartition() {
        return this.backgroundTaskServer.getPartition();
    }

    private boolean isTaskDeletedWhileProcessing(Exception e) {
        return Exceptions.hasCause(e, InterruptedException.class) && this.task.hasState(StateName.DELETED);
    }

    private boolean isTaskServerStopped(Exception e) {
        return Exceptions.hasCause(e, InterruptedException.class) && !this.task.hasState(StateName.DELETED);
    }

    private boolean isTaskNotFoundException(Exception e) {
        return e instanceof TaskNotFoundException;
    }

    @VisibleFor(value="testing")
    static Exception unwrapException(Exception e) {
        if (e instanceof InvocationTargetException && e.getCause() instanceof Exception) {
            return (Exception)e.getCause();
        }
        return e;
    }
}

