/*
 * Decompiled with CFR 0.152.
 */
package cn.boboweike.carrot.server;

import cn.boboweike.carrot.server.BackgroundTaskServer;
import cn.boboweike.carrot.server.dashboard.CpuAllocationIrregularityNotification;
import cn.boboweike.carrot.server.dashboard.DashboardNotificationManager;
import cn.boboweike.carrot.storage.BackgroundTaskServerStatus;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.storage.ServerTimedOutException;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerZooKeeper
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServerZooKeeper.class);
    private final BackgroundTaskServer backgroundTaskServer;
    private final PartitionedStorageProvider storageProvider;
    private final DashboardNotificationManager dashboardNotificationManager;
    private final int timeoutInSeconds;
    private final int lockDurationInSeconds;
    private final String lockedBy;
    private Instant lastSignalAlive;
    private Instant lastServerTimeoutCheck;
    private AtomicInteger restartAttempts;
    private final AtomicInteger extendLockFailureCount;

    public ServerZooKeeper(BackgroundTaskServer backgroundTaskServer) {
        this.backgroundTaskServer = backgroundTaskServer;
        this.storageProvider = backgroundTaskServer.getStorageProvider();
        this.dashboardNotificationManager = backgroundTaskServer.getDashboardNotificationManager();
        this.timeoutInSeconds = backgroundTaskServer.getServerStatus().getPollIntervalInSeconds() * 4;
        this.lockDurationInSeconds = backgroundTaskServer.getServerStatus().getPollIntervalInSeconds() * 4;
        this.lockedBy = backgroundTaskServer.getServerStatus().getId().toString();
        this.lastSignalAlive = Instant.now();
        this.lastServerTimeoutCheck = Instant.now();
        this.restartAttempts = new AtomicInteger();
        this.extendLockFailureCount = new AtomicInteger();
    }

    @Override
    public void run() {
        if (this.backgroundTaskServer.isStoppingOrStopped()) {
            return;
        }
        try {
            if (this.backgroundTaskServer.isUnAnnounced()) {
                this.announceBackgroundTaskServer();
            } else {
                this.signalBackgroundTaskServerAliveAndDoZooKeeping();
            }
            this.acquireOrExtendPartition();
        }
        catch (Exception shouldNotHappen) {
            if (this.restartAttempts.incrementAndGet() < 3) {
                LOGGER.error("An unrecoverable error occurred, try next run. Restart attempt: " + this.restartAttempts + " out of 3", (Throwable)shouldNotHappen);
            }
            LOGGER.error("An unrecoverable error occurred, restart server.", (Throwable)shouldNotHappen);
            this.restartAttempts.set(0);
            this.backgroundTaskServer.setPartition(null);
            new Thread(this::resetServer).start();
        }
    }

    private void acquireOrExtendPartition() {
        Integer partition = this.backgroundTaskServer.getPartition();
        if (partition == null || partition == BackgroundTaskServer.NO_PARTITION) {
            this.acquirePartition();
        } else {
            this.extendPartition(partition);
        }
    }

    public synchronized void stop() {
        try {
            this.storageProvider.signalBackgroundTaskServerStopped(this.backgroundTaskServer.getServerStatus());
            Integer partition = this.backgroundTaskServer.getPartition();
            if (partition != null && partition != BackgroundTaskServer.NO_PARTITION) {
                this.storageProvider.unlockByPartition(partition);
                LOGGER.info("Carrot server {} unlocked partition {}", (Object)this.lockedBy, (Object)partition);
            }
        }
        catch (Exception e) {
            LOGGER.error("Error when signalling that BackgroundTaskServer stopped", (Throwable)e);
        }
    }

    private void announceBackgroundTaskServer() {
        BackgroundTaskServerStatus serverStatus = this.backgroundTaskServer.getServerStatus();
        this.storageProvider.announceBackgroundTaskServer(serverStatus);
        this.lastSignalAlive = serverStatus.getLastHeartbeat();
    }

    private void signalBackgroundTaskServerAliveAndDoZooKeeping() {
        try {
            this.signalBackgroundTaskServerAlive();
            this.deleteServersThatTimedOut();
        }
        catch (ServerTimedOutException e) {
            LOGGER.error("SEVERE ERROR - Server timed out while it's still alive. Are all servers using NTP and in the same timezone? Are you having long GC cycles? Resetting server...", (Throwable)e);
            new Thread(this::resetServer).start();
        }
    }

    private void signalBackgroundTaskServerAlive() {
        BackgroundTaskServerStatus serverStatus = this.backgroundTaskServer.getServerStatus();
        this.storageProvider.signalBackgroundTaskServerAlive(serverStatus);
        this.cpuAllocationIrregularity(this.lastSignalAlive, serverStatus.getLastHeartbeat()).ifPresent(amountOfSeconds -> this.dashboardNotificationManager.notify(new CpuAllocationIrregularityNotification((Integer)amountOfSeconds)));
        this.lastSignalAlive = serverStatus.getLastHeartbeat();
    }

    private void deleteServersThatTimedOut() {
        if (Instant.now().isAfter(this.lastServerTimeoutCheck.plusSeconds(this.timeoutInSeconds))) {
            Instant timedOutInstantUsingLastSignalAlive;
            Instant now = Instant.now();
            Instant defaultTimeoutInstant = now.minusSeconds(this.timeoutInSeconds);
            Instant timedOutInstant = ServerZooKeeper.min(defaultTimeoutInstant, timedOutInstantUsingLastSignalAlive = this.lastSignalAlive.minusMillis(500L));
            int amountOfServersThatTimedOut = this.storageProvider.removeTimedOutBackgroundTaskServers(timedOutInstant);
            if (amountOfServersThatTimedOut > 0) {
                LOGGER.info("Removed {} server(s) that timed out", (Object)amountOfServersThatTimedOut);
            }
            this.lastServerTimeoutCheck = now;
        }
    }

    private void extendPartition(Integer partition) {
        boolean extendSuccess = this.storageProvider.extendLockByPartition(partition, this.lockDurationInSeconds, this.lockedBy);
        if (extendSuccess) {
            this.extendLockFailureCount.set(0);
            LOGGER.info("Carrot server {} extended partition {} for {} seconds", new Object[]{this.lockedBy, partition, this.lockDurationInSeconds});
        } else if (this.extendLockFailureCount.incrementAndGet() == 3) {
            this.backgroundTaskServer.setPartition(BackgroundTaskServer.NO_PARTITION);
            LOGGER.info("Carrot server {} failed to extend partition {} for {} seconds", new Object[]{this.lockedBy, partition, this.lockDurationInSeconds});
            this.extendLockFailureCount.set(0);
        }
    }

    private void acquirePartition() {
        List<Integer> partitionList = this.getShuffledPartitionList(this.storageProvider.getTotalNumOfPartitions());
        for (Integer partition : partitionList) {
            boolean lockSuccess = this.storageProvider.lockByPartition(partition, this.lockDurationInSeconds, this.lockedBy);
            if (lockSuccess) {
                LOGGER.info("Carrot server {} acquired partition {} for {} seconds", new Object[]{this.lockedBy, partition, this.lockDurationInSeconds});
                this.backgroundTaskServer.setPartition(partition);
                return;
            }
            this.sleepRandomly(100, 200);
        }
        LOGGER.info("Carrot server {} failed to acquire partition for {} seconds", (Object)this.lockedBy, (Object)this.lockDurationInSeconds);
        this.backgroundTaskServer.setPartition(BackgroundTaskServer.NO_PARTITION);
    }

    private void sleepRandomly(int low, int high) {
        try {
            long timeout = new Random().nextInt(high - low) + low;
            TimeUnit.MILLISECONDS.sleep(timeout);
        }
        catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    private List<Integer> getShuffledPartitionList(int totalNumberOfPartitions) {
        List<Integer> partitionList = IntStream.range(0, totalNumberOfPartitions).boxed().collect(Collectors.toList());
        Collections.shuffle(partitionList);
        return partitionList;
    }

    private void resetServer() {
        this.backgroundTaskServer.stop();
        this.backgroundTaskServer.start();
    }

    private static Instant min(Instant instant1, Instant instant2) {
        Object[] instants = new Instant[]{instant1, instant2};
        Arrays.sort(instants);
        return instants[0];
    }

    private Optional<Integer> cpuAllocationIrregularity(Instant lastSignalAlive, Instant lastHeartbeat) {
        int amount3OfSec;
        int amount2OfSec;
        Instant now = Instant.now();
        int amount1OfSec = (int)Math.abs(lastHeartbeat.getEpochSecond() - lastSignalAlive.getEpochSecond());
        int max = Math.max(amount1OfSec, Math.max(amount2OfSec = (int)(now.getEpochSecond() - lastSignalAlive.getEpochSecond()), amount3OfSec = (int)(now.getEpochSecond() - lastHeartbeat.getEpochSecond())));
        if ((long)max > (long)this.backgroundTaskServer.getServerStatus().getPollIntervalInSeconds() * 2L) {
            return Optional.of(max);
        }
        return Optional.empty();
    }
}

