package cn.com.anysdk.core.log;

import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;

/**
 * 操作日志工具类
 * 用于记录各种操作的日志，支持 MDC 上下文管理
 */
@Slf4j
public class OperationLogger {
    private static final String OPERATION_TYPE = "operationType";
    private static final String SERVICE = "service";
    private static final String PATH = "path";
    private static final String DURATION = "duration";
    private static final String SIZE = "size";
    private static final String IS_PRIVATE = "isPrivate";
    private static final String REQUEST_ID = "requestId";

    /**
     * 记录操作开始
     * @param methodName 方法名称
     * @param module 模块名称
     * @param description 描述
     */
    public static void logOperationStart(String methodName, String module, String description) {
        log.info("Operation started - Method: {}, Module: {}, Description: {}", methodName, module, description);
    }

    /**
     * 记录操作开始
     * @param methodName 方法名称
     * @param module 模块名称
     * @param description 描述
     * @param params 参数
     */
    public static void logOperationStart(String methodName, String module, String description, String params) {
        log.info("Operation started - Method: {}, Module: {}, Description: {}, Params: {}",
                methodName, module, description, params);
    }

    /**
     * 记录操作结束
     * @param duration 操作耗时（毫秒）
     * @param requestId 请求ID
     */
    public static void logOperationEnd(long duration, String requestId) {
        log.info("Operation completed - Duration: {}ms, RequestId: {}", duration, requestId);
    }

    /**
     * 记录操作结束
     * @param duration 操作耗时（毫秒）
     * @param result 结果
     * @param requestId 请求ID
     */
    public static void logOperationEnd(long duration, String result, String requestId) {
        log.info("Operation completed - Duration: {}ms, Result: {}, RequestId: {}", duration, result, requestId);
    }

    /**
     * 记录操作结束
     * @param duration 操作耗时（毫秒）
     * @param size 大小（字节）
     * @param isPrivate 是否为私有操作
     * @param requestId 请求ID
     */
    public static void logOperationEnd(long duration, long size, boolean isPrivate, String requestId) {
        MDC.put(DURATION, String.valueOf(duration));
        MDC.put(SIZE, String.valueOf(size));
        MDC.put(IS_PRIVATE, String.valueOf(isPrivate));
        MDC.put(REQUEST_ID, requestId);
        log.info("Operation completed - Duration: {}ms, Size: {}bytes, Private: {}, RequestId: {}",
                duration, size, isPrivate, requestId);
        clearContext();
    }

    /**
     * 记录操作错误
     * @param message 错误信息
     * @param e 异常对象
     */
    public static void logOperationError(String message, Throwable e) {
        log.error("Operation failed - Error: {}", message, e);
    }

    /**
     * 清除日志上下文
     */
    private static void clearContext() {
        MDC.remove(OPERATION_TYPE);
        MDC.remove(SERVICE);
        MDC.remove(PATH);
        MDC.remove(DURATION);
        MDC.remove(SIZE);
        MDC.remove(IS_PRIVATE);
        MDC.remove(REQUEST_ID);
    }
}