package cn.com.anysdk.oss.api;

import lombok.Data;
import lombok.experimental.Accessors;

/**
 * 上传选项
 * 用于配置文件上传的额外选项
 */
@Data
@Accessors(chain = true)
public class UploadOptions {
    /**
     * 是否为私有文件
     */
    private boolean isPrivate;

    /**
     * 过期时间（秒）
     */
    private Long expireTime;

    /**
     * 文件类型
     */
    private String contentType;

    /**
     * 访问控制权限
     */
    private String acl;

    /**
     * 文件大小限制（字节）
     */
    private Long maxSize;

    /**
     * 是否使用分片上传
     */
    private boolean useMultipart;

    /**
     * 分片大小（字节），默认5MB
     */
    private long partSize = 5 * 1024 * 1024;

    /**
     * 创建公开访问的上传选项
     * @return 上传选项
     */
    public static UploadOptions publicFile() {
        return new UploadOptions()
                .setPrivate(false)
                .setUseMultipart(false);
    }

    /**
     * 创建私有访问的上传选项
     * @param expireTime 过期时间（秒）
     * @return 上传选项
     */
    public static UploadOptions privateFile(Long expireTime) {
        return new UploadOptions()
                .setPrivate(true)
                .setExpireTime(expireTime)
                .setUseMultipart(false);
    }

    /**
     * 创建大文件上传选项
     * @param isPrivate 是否为私有文件
     * @param partSize 分片大小（字节）
     * @return 上传选项
     */
    public static UploadOptions largeFile(boolean isPrivate, long partSize) {
        return new UploadOptions()
                .setPrivate(isPrivate)
                .setUseMultipart(true)
                .setPartSize(partSize);
    }

    /**
     * 创建带文件类型的上传选项
     * @param contentType 文件类型
     * @param isPrivate 是否为私有文件
     * @return 上传选项
     */
    public static UploadOptions withContentType(String contentType, boolean isPrivate) {
        return new UploadOptions()
                .setPrivate(isPrivate)
                .setContentType(contentType)
                .setUseMultipart(false);
    }

    /**
     * 创建带访问控制的上传选项
     * @param acl 访问控制权限
     * @param isPrivate 是否为私有文件
     * @return 上传选项
     */
    public static UploadOptions withAcl(String acl, boolean isPrivate) {
        return new UploadOptions()
                .setPrivate(isPrivate)
                .setAcl(acl)
                .setUseMultipart(false);
    }
}