package cn.com.anysdk.oss.factory;

import cn.com.anysdk.oss.api.IOssService;
import cn.com.anysdk.oss.api.IOssConfig;
import cn.com.anysdk.oss.exception.OssException;
import cn.com.anysdk.oss.impl.AliyunOssService;
import cn.com.anysdk.oss.impl.LocalFileSystemService;
import cn.com.anysdk.oss.impl.MinioOssService;
import cn.com.anysdk.oss.impl.TencentCloudOssService;
import lombok.extern.slf4j.Slf4j;

/**
 * OSS服务工厂
 * 用于创建不同类型的OSS服务实例
 */
@Slf4j
public class OssServiceFactory {
    /**
     * 创建OSS服务实例
     * @param config OSS配置
     * @return OSS服务实例
     */
    public static IOssService createService(IOssConfig config) {
        if (config == null) {
            log.error("OSS config is null. Cannot create service.");
            throw new OssException("OSS config cannot be null");
        }

        String provider = config.getProvider();
        log.info("Creating OSS service for provider: {}", provider);

        try {
            return switch (provider.toLowerCase()) {
                case "aliyun" -> {
                    log.info("Initializing Aliyun OSS service.");
                    yield new AliyunOssService(config);
                }
                case "local" -> {
                    log.info("Initializing Local File System service.");
                    yield new LocalFileSystemService(config);
                }
                case "tencent" -> {
                    log.info("Initializing Tencent Cloud OSS service.");
                    yield new TencentCloudOssService(config);
                }
                case "minio" -> {
                    log.info("Initializing Minio Oss service.");
                    yield new MinioOssService(config);
                }
                default -> {
                    log.error("Unsupported OSS provider: {}", provider);
                    throw new OssException("Unsupported OSS provider: " + provider);
                }
            };
        } catch (Exception e) {
            log.error("Failed to create OSS service for provider: {}", provider, e);
            throw new OssException("Failed to create OSS service: " + e.getMessage(), e);
        }
    }
}
