package cn.com.anysdk.redis.event;

import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Redis事件管理器
 * 负责事件的注册和分发
 */
@Slf4j
public class RedisEventManager {

    private final List<RedisEventListener> listeners = new CopyOnWriteArrayList<>();

    /**
     * 注册事件监听器
     *
     * @param listener 事件监听器
     */
    public void registerListener(RedisEventListener listener) {
        if (listener != null && !listeners.contains(listener)) {
            listeners.add(listener);
            log.debug("注册Redis事件监听器: {}", listener.getClass().getName());
        }
    }

    /**
     * 移除事件监听器
     *
     * @param listener 事件监听器
     */
    public void removeListener(RedisEventListener listener) {
        if (listener != null) {
            listeners.remove(listener);
            log.debug("移除Redis事件监听器: {}", listener.getClass().getName());
        }
    }

    /**
     * 分发连接事件
     *
     * @param event 连接事件
     */
    public void fireConnectEvent(RedisConnectionEvent event) {
        for (RedisEventListener listener : listeners) {
            try {
                listener.onConnect(event);
            } catch (Exception e) {
                log.error("处理连接事件异常", e);
            }
        }
    }

    /**
     * 分发断开连接事件
     *
     * @param event 断开连接事件
     */
    public void fireDisconnectEvent(RedisConnectionEvent event) {
        for (RedisEventListener listener : listeners) {
            try {
                listener.onDisconnect(event);
            } catch (Exception e) {
                log.error("处理断开连接事件异常", e);
            }
        }
    }

    /**
     * 分发错误事件
     *
     * @param event 错误事件
     */
    public void fireErrorEvent(RedisErrorEvent event) {
        for (RedisEventListener listener : listeners) {
            try {
                listener.onError(event);
            } catch (Exception e) {
                log.error("处理错误事件异常", e);
            }
        }
    }

    /**
     * 分发重连事件
     *
     * @param event 重连事件
     */
    public void fireReconnectingEvent(RedisReconnectingEvent event) {
        for (RedisEventListener listener : listeners) {
            try {
                listener.onReconnecting(event);
            } catch (Exception e) {
                log.error("处理重连事件异常", e);
            }
        }
    }

    /**
     * 分发环境切换事件
     *
     * @param event 环境切换事件
     */
    public void fireEnvironmentChangeEvent(RedisEnvironmentChangeEvent event) {
        for (RedisEventListener listener : listeners) {
            try {
                listener.onEnvironmentChange(event);
            } catch (Exception e) {
                log.error("处理环境切换事件异常", e);
            }
        }
    }
}
