package cn.com.anysdk.redis.factory;

import cn.com.anysdk.redis.api.IRedisService;
import cn.com.anysdk.redis.config.RedisConfigProperties;
import cn.com.anysdk.redis.event.RedisEventManager;
import cn.com.anysdk.redis.impl.LocalRedisServiceImpl;
import cn.com.anysdk.redis.impl.AliyunRedisServiceImpl;
import lombok.extern.slf4j.Slf4j;

/**
 * Redis服务工厂
 * 根据环境配置创建对应的Redis服务实现
 */
@Slf4j
public class RedisServiceFactory {

    /**
     * 创建Redis服务实例
     * @param redisProperties Redis配置属性
     * @param eventManager 事件管理器
     * @return Redis服务实例
     */
    public static IRedisService createRedisService(RedisConfigProperties redisProperties,
                                                  RedisEventManager eventManager) {
        String environment = redisProperties.getEnvironment();
        log.info("创建Redis服务实例，环境: {}", environment);

        switch (environment.toLowerCase()) {
            case "local":
                return new LocalRedisServiceImpl(redisProperties, eventManager);
            case "aliyun":
                return new AliyunRedisServiceImpl(redisProperties, eventManager);
            case "huawei":
                // TODO: 实现华为云Redis服务
                throw new UnsupportedOperationException("华为云Redis服务暂未实现");
            case "tencent":
                // TODO: 实现腾讯云Redis服务
                throw new UnsupportedOperationException("腾讯云Redis服务暂未实现");
            default:
                log.warn("未知的Redis环境: {}，使用本地环境", environment);
                return new LocalRedisServiceImpl(redisProperties, eventManager);
        }
    }

    /**
     * 检查环境是否支持
     * @param environment 环境名称
     * @return 是否支持
     */
    public static boolean isSupportedEnvironment(String environment) {
        switch (environment.toLowerCase()) {
            case "local":
            case "aliyun":
                return true;
            case "huawei":
            case "tencent":
                return false; // 暂未实现
            default:
                return false;
        }
    }
}