package cn.com.anysdk.redis.reconnect;

/**
 * 指数退避重连策略
 * 每次重试的等待时间会按指数增长，直到达到最大等待时间
 */
public class ExponentialBackoffReconnectStrategy implements RedisReconnectStrategy {

    private final long initialIntervalMs;
    private final long maxIntervalMs;
    private final double multiplier;
    private final int maxRetryCount;

    /**
     * 创建指数退避重连策略
     *
     * @param initialIntervalMs 初始等待时间（毫秒）
     * @param maxIntervalMs 最大等待时间（毫秒）
     * @param multiplier 等待时间增长倍数
     * @param maxRetryCount 最大重试次数，-1表示无限重试
     */
    public ExponentialBackoffReconnectStrategy(long initialIntervalMs, long maxIntervalMs,
                                              double multiplier, int maxRetryCount) {
        this.initialIntervalMs = initialIntervalMs;
        this.maxIntervalMs = maxIntervalMs;
        this.multiplier = multiplier;
        this.maxRetryCount = maxRetryCount;
    }

    @Override
    public long getNextWaitTime(int retryCount) {
        // 计算当前重试的等待时间
        double waitTime = initialIntervalMs * Math.pow(multiplier, retryCount);

        // 确保不超过最大等待时间
        return Math.min((long) waitTime, maxIntervalMs);
    }

    @Override
    public int getMaxRetryCount() {
        return maxRetryCount;
    }

    @Override
    public void reset() {
        // 指数退避策略无需重置状态
    }

    @Override
    public String getName() {
        return "指数退避策略";
    }
}
