/*
 * Decompiled with CFR 0.152.
 */
package cn.com.anysdk.redis.factory;

import cn.com.anysdk.redis.config.AliyunRedisConfig;
import cn.com.anysdk.redis.config.RedisConfigProperties;
import cn.com.anysdk.redis.event.RedisConnectionEvent;
import cn.com.anysdk.redis.event.RedisEnvironmentChangeEvent;
import cn.com.anysdk.redis.event.RedisEventManager;
import cn.com.anysdk.redis.exception.AliyunRedisException;
import cn.com.anysdk.redis.exception.RedisConnectionException;
import cn.com.anysdk.redis.exception.RedisEnvironmentException;
import cn.com.anysdk.redis.exception.RedisExceptionHandler;
import cn.com.anysdk.redis.reconnect.ExponentialBackoffReconnectStrategy;
import cn.com.anysdk.redis.reconnect.RedisReconnectManager;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.JedisSentinelPool;

public class RedisConnectionFactory {
    private static final Logger log = LoggerFactory.getLogger(RedisConnectionFactory.class);
    private final RedisConfigProperties properties;
    private final RedisEventManager eventManager;
    private final RedisExceptionHandler exceptionHandler;
    private final RedisReconnectManager reconnectManager;
    private JedisPool jedisPool;
    private JedisCluster jedisCluster;
    private JedisSentinelPool jedisSentinelPool;
    private final AtomicBoolean initialized = new AtomicBoolean(false);

    public RedisConnectionFactory(RedisConfigProperties properties, RedisEventManager eventManager) {
        this.properties = properties;
        this.eventManager = eventManager;
        this.exceptionHandler = new RedisExceptionHandler(eventManager);
        ExponentialBackoffReconnectStrategy reconnectStrategy = new ExponentialBackoffReconnectStrategy(1000L, 30000L, 2.0, 10);
        this.reconnectManager = new RedisReconnectManager(eventManager, reconnectStrategy);
        this.initConnection();
    }

    public void initConnection() {
        try {
            JedisPoolConfig poolConfig = this.createPoolConfig();
            if ("local".equals(this.properties.getEnvironment())) {
                this.initLocalConnection(poolConfig);
            } else if ("aliyun".equals(this.properties.getEnvironment())) {
                this.initAliyunConnection(poolConfig);
            } else {
                throw new RedisEnvironmentException.UndefinedEnvironmentException("\u672a\u5b9a\u4e49\u7684Redis\u73af\u5883: " + this.properties.getEnvironment());
            }
            this.initialized.set(true);
            this.fireConnectionEvent(true);
            log.info("Redis\u8fde\u63a5\u521d\u59cb\u5316\u6210\u529f\uff0c\u5f53\u524d\u73af\u5883: {}", (Object)this.properties.getEnvironment());
        }
        catch (Exception e) {
            log.error("Redis\u8fde\u63a5\u521d\u59cb\u5316\u5931\u8d25", (Throwable)e);
            try {
                this.exceptionHandler.handleException(e, this, this.properties.getEnvironment());
            }
            catch (Exception ex) {
                this.reconnectManager.startReconnect(this, this.properties.getEnvironment(), this::reconnect);
            }
        }
    }

    private JedisPoolConfig createPoolConfig() {
        JedisPoolConfig poolConfig = new JedisPoolConfig();
        RedisConfigProperties.PoolConfig pool = "local".equals(this.properties.getEnvironment()) ? this.properties.getLocal().getPool() : this.properties.getAliyun().getPool();
        poolConfig.setMaxTotal(pool.getMaxTotal());
        poolConfig.setMaxIdle(pool.getMaxIdle());
        poolConfig.setMinIdle(pool.getMinIdle());
        poolConfig.setMaxWaitMillis(pool.getMaxWaitMillis());
        poolConfig.setTestOnBorrow(true);
        poolConfig.setTestOnReturn(true);
        poolConfig.setTestWhileIdle(true);
        poolConfig.setTimeBetweenEvictionRunsMillis(30000L);
        poolConfig.setNumTestsPerEvictionRun(3);
        poolConfig.setMinEvictableIdleTimeMillis(60000L);
        poolConfig.setBlockWhenExhausted(true);
        poolConfig.setJmxEnabled(false);
        return poolConfig;
    }

    private void initLocalConnection(JedisPoolConfig poolConfig) {
        this.closeAllConnections();
        RedisConfigProperties.LocalConfig local = this.properties.getLocal();
        this.jedisPool = local.getPassword() == null || local.getPassword().trim().isEmpty() || "null".equals(local.getPassword()) ? new JedisPool((GenericObjectPoolConfig)poolConfig, local.getHost(), local.getPort(), local.getTimeout().getConnectionTimeout(), null, local.getDatabase()) : new JedisPool((GenericObjectPoolConfig)poolConfig, local.getHost(), local.getPort(), local.getTimeout().getConnectionTimeout(), local.getPassword(), local.getDatabase());
        if (local.getPool().isPreWarm()) {
            this.preWarmConnections(local.getPool().getMinIdle());
        }
    }

    private void initAliyunConnection(JedisPoolConfig poolConfig) {
        if ("local".equals(this.properties.getEnvironment())) {
            return;
        }
        this.closeAllConnections();
        RedisConfigProperties.AliyunConfig aliyun = this.properties.getAliyun();
        this.jedisPool = new JedisPool((GenericObjectPoolConfig)poolConfig, "localhost", 6379, aliyun.getTimeout().getConnectionTimeout(), null, 0);
    }

    private void initAliyunSentinelConnection(JedisPoolConfig poolConfig, RedisConfigProperties.AliyunConfig aliyun, AliyunRedisConfig aliyunConfig) {
        String[] sentinelNodes = aliyunConfig.getSentinelNodes();
        if (sentinelNodes == null || sentinelNodes.length == 0) {
            throw new AliyunRedisException("\u963f\u91cc\u4e91Redis\u54e8\u5175\u8282\u70b9\u914d\u7f6e\u7f3a\u5931");
        }
        if (aliyunConfig.getMasterName() == null || aliyunConfig.getMasterName().isEmpty()) {
            throw new AliyunRedisException("\u963f\u91cc\u4e91Redis\u54e8\u5175\u4e3b\u8282\u70b9\u540d\u79f0\u914d\u7f6e\u7f3a\u5931");
        }
        HashSet<String> sentinels = new HashSet<String>(Arrays.asList(sentinelNodes));
        this.jedisSentinelPool = new JedisSentinelPool(aliyunConfig.getMasterName(), sentinels, (GenericObjectPoolConfig)poolConfig, aliyun.getTimeout().getConnectionTimeout(), aliyunConfig.getPassword(), aliyunConfig.getDatabase());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void preWarmConnections(int minConnections) {
        log.info("\u9884\u70edRedis\u8fde\u63a5\u6c60\uff0c\u521b\u5efa{}\u4e2a\u521d\u59cb\u8fde\u63a5", (Object)minConnections);
        HashSet<Jedis> connections = new HashSet<Jedis>();
        try {
            for (int i = 0; i < minConnections; ++i) {
                Jedis jedis = this.getConnection();
                String response = jedis.ping();
                if (!"PONG".equals(response)) {
                    throw new RedisConnectionException("Redis\u8fde\u63a5\u6d4b\u8bd5\u5931\u8d25\uff0c\u8fd4\u56de: " + response);
                }
                connections.add(jedis);
            }
        }
        finally {
            for (Jedis jedis : connections) {
                jedis.close();
            }
        }
    }

    public Jedis getConnection() {
        if (!this.initialized.get()) {
            throw new RedisConnectionException("Redis\u8fde\u63a5\u5c1a\u672a\u521d\u59cb\u5316");
        }
        return this.getConnectionWithRetry(3);
    }

    private Jedis getConnectionWithRetry(int maxRetries) {
        Exception lastException = null;
        for (int attempt = 1; attempt <= maxRetries; ++attempt) {
            try {
                Jedis jedis = this.jedisPool.getResource();
                if (this.isConnectionHealthy(jedis)) {
                    return jedis;
                }
                log.warn("\u83b7\u53d6\u5230\u4e0d\u5065\u5eb7\u7684Redis\u8fde\u63a5\uff0c\u5c1d\u8bd5\u91cd\u8bd5 ({}/{})", (Object)attempt, (Object)maxRetries);
                try {
                    jedis.close();
                }
                catch (Exception e) {
                    log.debug("\u5173\u95ed\u4e0d\u5065\u5eb7\u8fde\u63a5\u65f6\u51fa\u9519", (Throwable)e);
                }
                if (attempt >= maxRetries) continue;
                try {
                    Thread.sleep(100 * attempt);
                    continue;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RedisConnectionException("\u83b7\u53d6\u8fde\u63a5\u88ab\u4e2d\u65ad", e);
                }
            }
            catch (Exception e) {
                lastException = e;
                log.warn("\u83b7\u53d6Redis\u8fde\u63a5\u5931\u8d25\uff0c\u5c1d\u8bd5\u91cd\u8bd5 ({}/{}): {}", new Object[]{attempt, maxRetries, e.getMessage()});
                if (attempt >= maxRetries) continue;
                try {
                    Thread.sleep(100 * attempt);
                    continue;
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new RedisConnectionException("\u83b7\u53d6\u8fde\u63a5\u88ab\u4e2d\u65ad", ie);
                }
            }
        }
        log.error("\u83b7\u53d6Redis\u8fde\u63a5\u5931\u8d25\uff0c\u5df2\u91cd\u8bd5{}\u6b21", (Object)maxRetries);
        try {
            this.exceptionHandler.handleException(lastException, this, this.properties.getEnvironment());
        }
        catch (Exception ex) {
            this.reconnectManager.startReconnect(this, this.properties.getEnvironment(), this::reconnect);
        }
        throw new RedisConnectionException("\u83b7\u53d6Redis\u8fde\u63a5\u5931\u8d25: " + (lastException != null ? lastException.getMessage() : "\u672a\u77e5\u9519\u8bef"), lastException);
    }

    private boolean isConnectionHealthy(Jedis jedis) {
        if (jedis == null) {
            return false;
        }
        try {
            String response = jedis.ping();
            return "PONG".equals(response);
        }
        catch (Exception e) {
            log.debug("\u8fde\u63a5\u5065\u5eb7\u68c0\u67e5\u5931\u8d25: {}", (Object)e.getMessage());
            return false;
        }
    }

    public JedisCluster getClusterConnection() {
        if (!this.initialized.get()) {
            throw new RedisConnectionException("Redis\u8fde\u63a5\u5c1a\u672a\u521d\u59cb\u5316");
        }
        if (this.jedisCluster == null) {
            throw new RedisConnectionException("Redis\u96c6\u7fa4\u8fde\u63a5\u4e0d\u53ef\u7528\uff0c\u5f53\u524d\u6a21\u5f0f\u4e0d\u662f\u96c6\u7fa4\u6a21\u5f0f");
        }
        return this.jedisCluster;
    }

    private boolean reconnect() {
        try {
            log.info("\u5c1d\u8bd5\u91cd\u65b0\u8fde\u63a5Redis...");
            this.initConnection();
            return this.initialized.get();
        }
        catch (Exception e) {
            log.error("Redis\u91cd\u8fde\u5931\u8d25", (Throwable)e);
            return false;
        }
    }

    private void closeAllConnections() {
        if (this.jedisPool != null) {
            this.jedisPool.close();
            this.jedisPool = null;
        }
        if (this.jedisCluster != null) {
            try {
                this.jedisCluster.close();
            }
            catch (Exception e) {
                log.error("\u5173\u95edRedis\u96c6\u7fa4\u8fde\u63a5\u5931\u8d25", (Throwable)e);
            }
            this.jedisCluster = null;
        }
        if (this.jedisSentinelPool != null) {
            this.jedisSentinelPool.close();
            this.jedisSentinelPool = null;
        }
        if (this.initialized.getAndSet(false)) {
            this.fireConnectionEvent(false);
        }
    }

    public void close() {
        this.reconnectManager.stopReconnect();
        this.closeAllConnections();
        this.reconnectManager.shutdown();
    }

    public void switchEnvironment(String environment) {
        if (environment == null || environment.isEmpty()) {
            throw new RedisEnvironmentException("\u73af\u5883\u540d\u79f0\u4e0d\u80fd\u4e3a\u7a7a");
        }
        if (!"local".equals(environment) && !"aliyun".equals(environment)) {
            throw new RedisEnvironmentException.UndefinedEnvironmentException("\u672a\u5b9a\u4e49\u7684Redis\u73af\u5883: " + environment);
        }
        if (environment.equals(this.properties.getEnvironment())) {
            log.info("\u5f53\u524d\u5df2\u7ecf\u662f{}\u73af\u5883\uff0c\u65e0\u9700\u5207\u6362", (Object)environment);
            return;
        }
        String oldEnvironment = this.properties.getEnvironment();
        try {
            this.reconnectManager.stopReconnect();
            this.closeAllConnections();
            this.properties.setEnvironment(environment);
            this.initConnection();
            this.eventManager.fireEnvironmentChangeEvent(new RedisEnvironmentChangeEvent(this, environment, oldEnvironment, environment, new Date()));
            log.info("Redis\u73af\u5883\u5df2\u4ece{}\u5207\u6362\u5230{}", (Object)oldEnvironment, (Object)environment);
        }
        catch (Exception e) {
            this.properties.setEnvironment(oldEnvironment);
            log.error("Redis\u73af\u5883\u5207\u6362\u5931\u8d25", (Throwable)e);
            try {
                this.exceptionHandler.handleException(new RedisEnvironmentException.SwitchException("\u73af\u5883\u5207\u6362\u5931\u8d25: " + e.getMessage(), e), this, oldEnvironment);
            }
            catch (Exception ex) {
                this.reconnectManager.startReconnect(this, oldEnvironment, this::reconnect);
            }
        }
    }

    private void fireConnectionEvent(boolean connected) {
        String host = this.properties.getLocal().getHost();
        int port = this.properties.getLocal().getPort();
        RedisConnectionEvent event = new RedisConnectionEvent(this, this.properties.getEnvironment(), host, port, new Date());
        if (connected) {
            this.eventManager.fireConnectEvent(event);
        } else {
            this.eventManager.fireDisconnectEvent(event);
        }
    }
}

