package cn.com.anysdk.sms.impl;

import cn.com.anysdk.sms.api.ISmsConfig;
import cn.com.anysdk.sms.api.ISmsService;
import cn.com.anysdk.sms.api.SmsRequest;
import cn.com.anysdk.sms.api.SmsResponse;
import cn.com.anysdk.sms.config.MockSmsConfig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 模拟短信服务实现类，用于测试环境
 * 不会真正发送短信，但会模拟短信发送的行为和响应
 */
public class MockSmsService implements ISmsService {
    
    // 存储发送的短信记录，key为messageId
    private final Map<String, MockSmsRecord> smsRecords = new ConcurrentHashMap<>();
    
    // 模拟的SMS配置
    private final MockSmsConfig config;
    
    public MockSmsService() {
        this.config = new MockSmsConfig();
    }
    
    public MockSmsService(MockSmsConfig config) {
        this.config = config;
    }

    @Override
    public SmsResponse send(SmsRequest request) {
        // 生成唯一的消息ID
        String messageId = UUID.randomUUID().toString();

        // 创建模拟的短信记录
        MockSmsRecord record = new MockSmsRecord(
                messageId,
                request.getPhoneNumber(),
                request.getTemplateId(),
                request.getExtraParams(),
                System.currentTimeMillis()
        );

        // 存储短信记录
        smsRecords.put(messageId, record);

        // 构建响应，补充更多字段
        return SmsResponse.builder()
                .success(true)
                .messageId(messageId)
                .status("DELIVERED")
                .extraData(request.getExtraParams())
                .build();
    }

    @Override
    public List<SmsResponse> sendBatch(List<SmsRequest> requests) {
        List<SmsResponse> responses = new ArrayList<>();
        for (SmsRequest request : requests) {
            responses.add(send(request));
        }
        return responses;
    }

    @Override
    public SmsResponse queryStatus(String messageId) {
        MockSmsRecord record = smsRecords.get(messageId);
        if (record != null) {
            return SmsResponse.builder()
                    .success(true)
                    .messageId(messageId)
                    .status("DELIVERED")
                    .sendTime(null)
                    .receiveTime(null)
                    .extraData(record.getExtraParams())
                    .build();
        } else {
            return SmsResponse.builder()
                    .success(false)
                    .messageId(messageId)
                    .status("NOT_FOUND")
                    .errorCode("404")
                    .errorMessage("短信记录不存在")
                    .build();
        }
    }

    @Override
    public SmsResponse sendWithExtra(SmsRequest request, Map<String, Object> extraParams) {
        // 将额外参数添加到记录中
        SmsResponse response = send(request);
        if (response.isSuccess() && extraParams != null) {
            MockSmsRecord record = smsRecords.get(response.getMessageId());
            if (record != null) {
                record.setExtraParams(new HashMap<>(extraParams));
            }
        }
        return response;
    }

    @Override
    public List<SmsResponse> queryStatusBatch(List<String> messageIds) {
        List<SmsResponse> responses = new ArrayList<>();
        for (String messageId : messageIds) {
            responses.add(queryStatus(messageId));
        }
        return responses;
    }

    @Override
    public SmsResponse getQuota() {
        Map<String, Object> quotaInfo = new HashMap<>();
        quotaInfo.put("dailyLimit", 10000);
        quotaInfo.put("dailyUsed", 0);
        quotaInfo.put("monthlyLimit", 300000);
        quotaInfo.put("monthlyUsed", 0);

        return SmsResponse.builder()
                .success(true)
                .extraData(quotaInfo)
                .build();
    }

    @Override
    public String getProviderType() {
        return "MOCK";
    }

    @Override
    public ISmsConfig getConfig() {
        return this.config;
    }
    
    /**
     * 获取所有已发送的短信记录
     * @return 短信记录Map
     */
    public Map<String, MockSmsRecord> getAllSmsRecords() {
        return new HashMap<>(smsRecords);
    }
    
    /**
     * 清除所有短信记录
     */
    public void clearAllRecords() {
        smsRecords.clear();
    }
    
    /**
     * 模拟短信记录内部类
     */
    public static class MockSmsRecord {
        private final String messageId;
        private final String phoneNumber;
        private final String templateId;
        private final Map<String, Object> params;
        private final long sendTime;
        private Map<String, Object> extraParams;
        
        public MockSmsRecord(String messageId, String phoneNumber, String templateId,
                             Map<String, Object> params, long sendTime) {
            this.messageId = messageId;
            this.phoneNumber = phoneNumber;
            this.templateId = templateId;
            this.params = params;
            this.sendTime = sendTime;
        }
        
        // Getters
        public String getMessageId() {
            return messageId;
        }
        
        public String getPhoneNumber() {
            return phoneNumber;
        }
        
        public String getTemplateId() {
            return templateId;
        }
        
        public Map<String, Object> getParams() {
            return params;
        }
        
        public long getSendTime() {
            return sendTime;
        }
        
        public Map<String, Object> getExtraParams() {
            return extraParams;
        }
        
        public void setExtraParams(Map<String, Object> extraParams) {
            this.extraParams = extraParams;
        }
    }
}

