package cn.com.dabby.idaas.oidc;

import cn.com.dabby.idaas.oidc.exception.OAuth2Exception;
import cn.com.dabby.idaas.oidc.pojo.IdTokenClaim;
import cn.com.dabby.idaas.oidc.pojo.OidcTokenResponse;
import cn.com.dabby.idaas.oidc.pojo.OidcUserInfo;
import cn.com.dabby.idaas.oidc.utils.Asserts;
import cn.com.dabby.idaas.oidc.utils.JsonUtils;
import cn.com.dabby.idaas.oidc.utils.StringUtils;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.oauth2.sdk.*;
import com.nimbusds.oauth2.sdk.auth.ClientAuthentication;
import com.nimbusds.oauth2.sdk.auth.ClientSecretBasic;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.Issuer;
import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;
import com.nimbusds.openid.connect.sdk.*;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;
import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;
import com.nimbusds.openid.connect.sdk.validators.IDTokenValidator;

import java.net.URI;

public class OidcClient {
    private static final Scope DEFAULT_SCOPE = new Scope("openid", "profile", "user_bz", "menu", "role", "permission");
    private static final ResponseType DEFAULT_RESPONSE_TYPE = new ResponseType("code");

    private final OIDCProviderMetadata oidcProviderMetadata;
    private final ClientID clientId;
    private final Secret clientSecret;
    private final URI redirectUri;
    private final Issuer issuer;
    private JWKSet jwkSet;
    private final IDTokenValidator idTokenValidator;

    public OidcClient(String issuer, String clientid, String clientSecret, String redirectUri) throws Exception {
        Asserts.hasText(issuer, "issuer 不能为空");
        Asserts.hasText(clientid, "clientId 不能为空");
        Asserts.hasText(clientSecret, "clientSecret 不能为空");
        Asserts.hasText(redirectUri, "redirectUri 不能为空");

        this.issuer = new Issuer(issuer);
        this.oidcProviderMetadata = OIDCProviderMetadata.resolve(this.issuer);

        this.jwkSet = JWKSet.load(oidcProviderMetadata.getJWKSetURI().toURL());
        this.clientId = new ClientID(clientid);
        this.clientSecret = new Secret(clientSecret);
        this.redirectUri = new URI(redirectUri);
        this.idTokenValidator = new IDTokenValidator(this.issuer, this.clientId, JWSAlgorithm.RS256, jwkSet);
    }

    public OIDCProviderMetadata getOidcProviderMetadata() {
        return oidcProviderMetadata;
    }

    public String buildAuthorizationUrl(String state, String nonce) {
        return new AuthenticationRequest.Builder(
                DEFAULT_RESPONSE_TYPE,
                DEFAULT_SCOPE,
                clientId,
                redirectUri)
                .endpointURI(oidcProviderMetadata.getAuthorizationEndpointURI())
                .state(new State(state))
                .nonce(new Nonce(nonce))
                .build()
                .toURI()
                .toASCIIString();
    }

    public OidcTokenResponse getTokenByCode(String authorizationCode) throws Exception {
        AuthorizationCode code = new AuthorizationCode(authorizationCode);
        AuthorizationGrant codeGrant = new AuthorizationCodeGrant(code, redirectUri);
        ClientAuthentication clientAuth = new ClientSecretBasic(clientId, clientSecret);
        URI tokenEndpoint = oidcProviderMetadata.getTokenEndpointURI();

        TokenRequest request = new TokenRequest(tokenEndpoint, clientAuth, codeGrant);

        TokenResponse tokenResponse = OIDCTokenResponseParser.parse(request.toHTTPRequest().send());

        if (!tokenResponse.indicatesSuccess()) {
            final ErrorObject errorObject = tokenResponse.toErrorResponse().getErrorObject();
            String errorCode = errorObject.getCode();
            String errorDescription = errorObject.getCustomParams().getOrDefault("errorDescription", "未知异常");
            throw new OAuth2Exception(errorCode, errorDescription);
        }

        String successResponseJson = tokenResponse.toSuccessResponse().getTokens().toJSONObject().toJSONString();

        return JsonUtils.fromJson(successResponseJson, OidcTokenResponse.class);
    }

    public IdTokenClaim validateIdToken(String idToken, String expectedNonce) throws Exception {
        final JWT jwt = JWTParser.parse(idToken);
        Nonce nonce = null;
        if (!StringUtils.isBlank(expectedNonce)) {
             nonce = new Nonce(expectedNonce);
        }

        final IDTokenClaimsSet idTokenClaims = idTokenValidator.validate(jwt, nonce);
        final String json = idTokenClaims.toJSONString();
        return JsonUtils.fromJson(json, IdTokenClaim.class);
    }

    public OidcUserInfo getUserInfo(String accessToken) throws Exception {
        URI userInfoEndpoint = oidcProviderMetadata.getUserInfoEndpointURI();
        BearerAccessToken token = new BearerAccessToken(accessToken);

        HTTPResponse httpResponse = new UserInfoRequest(userInfoEndpoint, token)
                .toHTTPRequest()
                .send();

        UserInfoResponse userInfoResponse = UserInfoResponse.parse(httpResponse);

        if (!userInfoResponse.indicatesSuccess()) {
            final ErrorObject errorObject = userInfoResponse.toErrorResponse().getErrorObject();
            String errorCode = errorObject.getCode();
            String errorDescription = errorObject.getCustomParams().getOrDefault("errorDescription", "未知异常");
            throw new OAuth2Exception(errorCode, errorDescription);
        }

        String userInfoJson = userInfoResponse.toSuccessResponse().getUserInfo().toJSONString();
        return JsonUtils.fromJson(userInfoJson, OidcUserInfo.class);
    }

}
