package cn.net.wanmo.common.compress;

import net.lingala.zip4j.ZipFile;
import net.lingala.zip4j.exception.ZipException;
import net.lingala.zip4j.model.ZipParameters;
import net.lingala.zip4j.model.enums.CompressionLevel;
import net.lingala.zip4j.model.enums.EncryptionMethod;

import java.io.File;
import java.io.IOException;
import java.util.List;

public class Zip4jUtil {

    /**
     * 压缩单个文件
     *
     * @param sourceFile  待压缩的文件
     * @param descZipFile 压缩的文件
     * @throws IOException 异常
     */
    public static void compressFile(File sourceFile, File descZipFile) throws IOException {
        compressFile(sourceFile, descZipFile, null);
    }

    /**
     * 压缩单个文件
     *
     * @param sourceFile  待压缩的文件
     * @param descZipFile 压缩的文件
     * @param password    密码
     * @throws IOException 异常
     */
    public static void compressFile(File sourceFile, File descZipFile, String password) throws IOException {
        ZipFile zipFile = getZipFile(descZipFile, password);
        ZipParameters zipParameters = getZipParameters(password);
        compressFile(sourceFile, zipFile, zipParameters);
    }

    /**
     * 压缩单个文件
     *
     * @param sourceFile    待压缩的文件
     * @param zipFile       压缩对象
     * @param zipParameters 压缩参数
     * @throws IOException 异常
     */
    public static void compressFile(File sourceFile, ZipFile zipFile, ZipParameters zipParameters) throws IOException {
        zipFile.addFile(sourceFile, zipParameters);
        zipFile.close();
    }

    /**
     * 压缩多个文件
     *
     * @param sourceFiles 待压缩的文件
     * @param descZipFile 压缩的文件
     * @throws IOException 异常
     */
    public static void compressFiles(List<File> sourceFiles, File descZipFile) throws IOException {
        compressFiles(sourceFiles, descZipFile, null);
    }

    /**
     * 压缩多个文件
     *
     * @param sourceFiles 待压缩的文件
     * @param descZipFile 压缩的文件
     * @param password    密码
     * @throws IOException 异常
     */
    public static void compressFiles(List<File> sourceFiles, File descZipFile, String password) throws IOException {
        ZipFile zipFile = getZipFile(descZipFile, password);
        ZipParameters zipParameters = getZipParameters(password);
        compressFiles(sourceFiles, zipFile, zipParameters);
    }

    /**
     * 压缩多个文件
     *
     * @param sourceFiles   待压缩的文件
     * @param zipFile       压缩对象
     * @param zipParameters 压缩参数
     * @throws IOException 异常
     */
    public static void compressFiles(List<File> sourceFiles, ZipFile zipFile, ZipParameters zipParameters) throws IOException {
        zipFile.addFiles(sourceFiles, zipParameters);
        zipFile.close();
    }

    /**
     * 压缩多个文件，并分割
     *
     * @param sourceFiles 待压缩的文件
     * @param descZipFile 压缩的文件
     * @param splitLength 分割单位是字节 1024 * 1024 * 10; //10MB
     * @throws IOException 异常
     */
    public static void compressFilesSplit(List<File> sourceFiles, File descZipFile, int splitLength) throws IOException {
        compressFilesSplit(sourceFiles, descZipFile, null, splitLength);
    }

    /**
     * 压缩多个文件，并分割
     *
     * @param sourceFiles 待压缩的文件
     * @param descZipFile 压缩的文件
     * @param password    密码
     * @param splitLength 分割单位是字节 1024 * 1024 * 10; //10MB
     * @throws IOException 异常
     */
    public static void compressFilesSplit(List<File> sourceFiles, File descZipFile, String password, int splitLength) throws IOException {
        ZipFile zipFile = getZipFile(descZipFile, password);
        ZipParameters zipParameters = getZipParameters(password);
        compressFilesSplit(sourceFiles, zipFile, zipParameters, splitLength);
    }

    /**
     * 压缩多个文件，并分割
     *
     * @param sourceFiles   待压缩的文件
     * @param zipFile       压缩对象
     * @param zipParameters 压缩参数
     * @param splitLength   分割单位是字节 1024 * 1024 * 10; //10MB
     * @throws IOException 异常
     */
    public static void compressFilesSplit(List<File> sourceFiles, ZipFile zipFile, ZipParameters zipParameters, int splitLength) throws IOException {
        zipFile.createSplitZipFile(sourceFiles, zipParameters, true, splitLength);
        zipFile.close();
    }

    /**
     * 压缩文件夹
     *
     * @param sourceDir   待压缩的文件夹
     * @param descZipFile 压缩的文件
     * @throws IOException 异常
     */
    public static void compressDir(File sourceDir, File descZipFile) throws IOException {
        compressDir(sourceDir, descZipFile, null);
    }

    /**
     * 压缩文件夹
     *
     * @param sourceDir   待压缩的文件夹
     * @param descZipFile 压缩的文件
     * @param password    密码
     * @throws IOException 异常
     */
    public static void compressDir(File sourceDir, File descZipFile, String password) throws IOException {
        ZipFile zipFile = getZipFile(descZipFile, password);
        ZipParameters zipParameters = getZipParameters(password);
        compressDir(sourceDir, zipFile, zipParameters);
    }

    /**
     * 压缩文件夹
     *
     * @param sourceDir     待压缩的文件夹
     * @param zipFile       压缩对象
     * @param zipParameters 压缩参数
     * @throws IOException 异常
     */
    public static void compressDir(File sourceDir, ZipFile zipFile, ZipParameters zipParameters) throws IOException {
        zipFile.addFolder(sourceDir, zipParameters);
        zipFile.close();
    }

    /**
     * 压缩文件夹，并分割
     *
     * @param sourceDir   待压缩的文件夹
     * @param descZipFile 压缩的文件
     * @param splitLength 分割单位是字节 1024 * 1024 * 10; //10MB
     * @throws IOException 异常
     */
    public static void compressDirSplit(File sourceDir, File descZipFile, int splitLength) throws IOException {
        compressDirSplit(sourceDir, descZipFile, null, splitLength);
    }

    /**
     * 压缩文件夹，并分割
     *
     * @param sourceDir   待压缩的文件夹
     * @param descZipFile 压缩的文件
     * @param password    密码
     * @param splitLength 分割单位是字节 1024 * 1024 * 10; //10MB
     * @throws IOException 异常
     */
    public static void compressDirSplit(File sourceDir, File descZipFile, String password, int splitLength) throws IOException {
        ZipFile zipFile = getZipFile(descZipFile, password);
        ZipParameters zipParameters = getZipParameters(password);
        compressDirSplit(sourceDir, zipFile, zipParameters, splitLength);
    }

    /**
     * 压缩文件夹，并分割
     *
     * @param sourceDir     待压缩的文件夹
     * @param zipFile       压缩对象
     * @param zipParameters 压缩参数
     * @param splitLength   分割单位是字节 1024 * 1024 * 10; //10MB
     * @throws IOException 异常
     */
    public static void compressDirSplit(File sourceDir, ZipFile zipFile, ZipParameters zipParameters, int splitLength) throws IOException {
        zipFile.createSplitZipFileFromFolder(sourceDir, zipParameters, true, splitLength);
        zipFile.close();
    }

    /**
     * 提取一个文件
     *
     * @param sourceZipFile   待解压的文件
     * @param fileName        要提取的文件名
     * @param destinationPath 解压路径
     * @throws ZipException 异常
     */
    public static void extractFile(String sourceZipFile, String fileName, String destinationPath) throws ZipException {
        extractFile(sourceZipFile, fileName, destinationPath, null);
    }

    /**
     * 提取一个文件
     *
     * @param sourceZipFile   待解压的文件
     * @param fileName        要提取的文件名
     * @param destinationPath 解压路径
     * @param password        密码
     * @throws ZipException 异常
     */
    public static void extractFile(String sourceZipFile, String fileName, String destinationPath, String password) throws ZipException {
        ZipFile zipFile;

        if (password == null || password.trim() == "") {
            zipFile = new ZipFile(sourceZipFile);
        } else {
            zipFile = new ZipFile(sourceZipFile, password.toCharArray());
        }

        zipFile.extractFile(fileName, destinationPath);
    }

    /**
     * 解压所有文件
     *
     * @param sourceZipFile   待解压的文件
     * @param destinationPath 解压路径
     */
    public static void extractAll(String sourceZipFile, String destinationPath) throws ZipException {
        extractAll(sourceZipFile, destinationPath, null);
    }

    /**
     * 解压所有文件
     *
     * @param sourceZipFile   待解压的文件
     * @param destinationPath 解压路径
     * @param password        密码
     */
    public static void extractAll(String sourceZipFile, String destinationPath, String password) throws ZipException {
        ZipFile zipFile;

        if (password == null || password.trim() == "") {
            zipFile = new ZipFile(sourceZipFile);
        } else {
            zipFile = new ZipFile(sourceZipFile, password.toCharArray());
        }

        zipFile.extractAll(destinationPath);
    }

    /**
     * 压缩文件
     *
     * @param descZipFile 文件
     * @param password    密码
     * @return 压缩对象
     */
    private static ZipFile getZipFile(File descZipFile, String password) {
        ZipFile zipFile;

        if (password == null || password.trim() == "") {
            zipFile = new ZipFile(descZipFile);
        } else {
            zipFile = new ZipFile(descZipFile, password.toCharArray());
        }

        return zipFile;
    }

    /**
     * 压缩参数
     *
     * @param password 密码
     * @return 压缩参数
     */
    private static ZipParameters getZipParameters(String password) {
        ZipParameters zipParameters = new ZipParameters();

        if (password == null || password.trim() == "") {
            zipParameters.setEncryptionMethod(EncryptionMethod.NONE);
        } else {
            zipParameters.setEncryptFiles(true);
            zipParameters.setCompressionLevel(CompressionLevel.HIGHER);
            zipParameters.setEncryptionMethod(EncryptionMethod.ZIP_STANDARD);
        }

        return zipParameters;
    }

}


