package cn.net.wanmo.common.compress;

import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.compress.utils.IOUtils;

import java.io.*;

public class ApacheUtil {
    /**
     * 使用 Apache Commons Compress 库进行Gzip压缩 <br/>
     * 注：压缩后文件后缀没有了
     * @param inputFile  要压缩的文件
     * @param outputFile 压缩后的文件
     * @throws IOException 压缩过程中可能出现的异常
     */
    public static void gzipCompress(File inputFile, File outputFile) throws IOException {
        // 创建输入流读取要压缩的文件
        try (FileInputStream fis = new FileInputStream(inputFile);
             // 创建压缩输出流写入压缩后的文件
             GzipCompressorOutputStream gzos = new GzipCompressorOutputStream(new FileOutputStream(outputFile))) {
            // 将输入流的数据压缩并写入输出流
            IOUtils.copy(fis, gzos);
        }
    }

    /**
     * 使用 Apache Commons Compress 库进行Gzip解压缩
     *
     * @param inputFile  压缩文件
     * @param outputFile 解压后的文件
     * @throws IOException 解压过程中可能出现的异常
     */
    public static void gzipDecompress(File inputFile, File outputFile) throws IOException {
        // 创建输入流读取压缩文件
        try (GzipCompressorInputStream gzis = new GzipCompressorInputStream(new FileInputStream(inputFile));
             // 创建输出流写入解压后的文件
             FileOutputStream fos = new FileOutputStream(outputFile)) {
            // 将输入流的数据解压并写入输出流
            IOUtils.copy(gzis, fos);
        }
    }

    /**
     * 使用 Apache Commons Compress 库进行Tar压缩
     *
     * @param inputFiles 要压缩的文件列表
     * @param outputFile 压缩后的文件
     * @throws IOException 压缩过程中可能出现的异常
     */
    public static void tarCompress(File[] inputFiles, File outputFile) throws IOException {
        // 创建压缩输出流写入压缩后的文件
        try (TarArchiveOutputStream taros = new TarArchiveOutputStream(new FileOutputStream(outputFile))) {
            // 设置压缩格式为gzip
            taros.setLongFileMode(TarArchiveOutputStream.LONGFILE_GNU);

            // 遍历要压缩的文件列表
            for (File inputFile : inputFiles) {
                // 创建压缩实体并设置文件名
                TarArchiveEntry entry = taros.createArchiveEntry(inputFile, inputFile.getName());
                // 将实体写入压缩输出流
                taros.putArchiveEntry(entry);

                // 创建输入流读取要压缩的文件
                try (FileInputStream fis = new FileInputStream(inputFile)) {
                    // 将输入流的数据写入压缩输出流
                    IOUtils.copy(fis, taros);
                }

                // 关闭当前压缩实体
                taros.closeArchiveEntry();
            }
        }
    }

    /**
     * 使用 Apache Commons Compress 库进行Tar解压缩
     *
     * @param inputFile   压缩文件
     * @param outputDir   解压后的目录，必须存在，否则报错
     * @throws IOException 解压过程中可能出现的异常
     */
    public static void tarDecompress(File inputFile, File outputDir) throws IOException {
        // 创建输入流读取压缩文件
        try (TarArchiveInputStream taris = new TarArchiveInputStream(new FileInputStream(inputFile))) {
            // 遍历压缩文件中的实体
            ArchiveEntry entry;
            while ((entry = taris.getNextEntry()) != null) {
                // 获取实体的文件名
                String fileName = entry.getName();
                // 创建输出文件并设置输出目录
                File outputFile = new File(outputDir, fileName);

                // 如果实体是目录，则创建相应目录；否则创建输出文件并写入数据
                if (entry.isDirectory()) {
                    outputFile.mkdirs();
                } else {
                    try (FileOutputStream fos = new FileOutputStream(outputFile)) {
                        // 将输入流的数据写入输出文件
                        IOUtils.copy(taris, fos);
                    }
                }
            }
        }
    }

    public static void main(String[] args) {
        try {
            File inputFile = new File("input.txt");
            File gzipFile = new File("output.gz");
            File outputFile = new File("output.txt");
            File[] tarInputFiles = {new File("file1.txt"), new File("file2.txt")};
            File tarFile = new File("output.tar");

            // Gzip压缩
            gzipCompress(inputFile, gzipFile);

            // Gzip解压缩
            gzipDecompress(gzipFile, outputFile);

            // Tar压缩
            tarCompress(tarInputFiles, tarFile);

            // Tar解压缩
            tarDecompress(tarFile, new File("outputDir"));
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}


