package cn.net.wanmo.common.ehcache2;

import java.io.Serializable;
import java.util.Set;
import net.sf.ehcache.Cache;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.search.Query;
import net.sf.ehcache.store.MemoryStoreEvictionPolicy;

/**
 * 缓存对象
 * @param <Key> 缓存 key 类型
 * @param <Value> 缓存 value 类型
 */
public class CacheObj<Key extends Serializable, Value extends Serializable> {
    private String cacheName;
    private Cache cache;

    public CacheObj(String cacheName) {
        this.cacheName = cacheName;

        Ehcache ehcache = CacheUtil.addCacheIfAbsent(cacheName);
        CacheConfiguration configuration = ehcache.getCacheConfiguration();
        configuration.memoryStoreEvictionPolicy(MemoryStoreEvictionPolicy.LRU); // 最少被使用的

        configuration.setMaxElementsInMemory(200); // 允许内存中最大的元素个数
        configuration.setOverflowToDisk(true); // 当元素的数量超过maxElementsInMemory的值时，是否允许存储到磁盘
        configuration.setDiskSpoolBufferSizeMB(100); // 磁盘缓存区大小，默认为30MB。每个Cache都应该有自己的一个缓存区。

        configuration.setDiskExpiryThreadIntervalSeconds(60 * 60 * 6); // 磁盘中数据过期检查线程的轮询时间间隔

        configuration.setEternal(false); // true时表示缓存中的数据永远不会过期；false时表示缓存中的数据会过期，过期的依据需要根据timeToIdleSeconds 和 timeToLiveSeconds 的值
        configuration.setTimeToIdleSeconds(60 * 60 * 6); // 允许缓存中数据的最大空闲时间，即cache.get(key)最近一次被调用离现在的时间间隔如果大于该时间，即该元素会被过期expiry
        configuration.setTimeToLiveSeconds(60 * 60 * 12); // 允许缓存中数据的最大生存时间，0时表示无限大。cache.put(element)到现在的时间间隔如果大于该时间，即该元素被过期expiry

        this.cache = CacheUtil.getCache(cacheName);
    }

    public Cache getCache() {
        return cache;
    }

    public void put(Key key, Value value) {
        Element element = new Element(key, value);
        cache.put(element);
    }

    public Element putIfAbsent(Key key, Value value) {
        Element element = new Element(key, value);
        return cache.putIfAbsent(element);
    }

    public Value get(Key key) {
        Element element = cache.get(key);
        Object value = element.getObjectValue();
        return cast(value);
    }

    public Integer getSize() {
        return cache == null ? null : cache.getSize();
    }

    public Query createQuery() {
        return cache.createQuery();
    }

    public void evictExpiredElements() {
        cache.evictExpiredElements();
    }

    public boolean remove(Key key) {
        return cache.remove(key);
    }

    public void remove(Set<Key> set) {
        cache.removeAll(set);
    }

    public Element replace(Key key, Value value) {
        Element element = new Element(key, value);
        return cache.replace(element);
    }

    public boolean replace(Key key, Value valueOld, Value valueNew) {
        Element elementOld = new Element(key, valueOld);
        Element elementNew = new Element(key, valueNew);
        return cache.replace(elementOld, elementNew);
    }

    public boolean isKeyInCache(Key key) {
        return cache.isKeyInCache(key);
    }

    public boolean isValueInCache(Key key) {
        return cache.isValueInCache(key);
    }

    public String getName() {
        return cache.getName();
    }

    public void flush() {
        cache.flush();
    }

    public void dispose() {
        cache.dispose();
    }

    private <T> T cast(Object obj) {
        return (T) obj;
    }

}
