package cn.net.wanmo.common.exec;

import org.apache.commons.exec.OS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

/**
 * 命令行执行工具类 <br/>
 * cmd /c dir：执行完dir命令后关闭命令窗口； <br/>
 * cmd /k dir：执行完dir命令后不关闭命令窗口。 <br/>
 * cmd /c start dir：会打开一个新窗口后执行dir指令，原窗口会关闭； <br/>
 * cmd /k start dir：会打开一个新窗口后执行dir指令，原窗口不会关闭。 <br/>
 */
public class RuntimeUtil {
    private static Logger logger = LoggerFactory.getLogger(RuntimeUtil.class);

    /**
     * 执行指令
     *
     * @param command 指令
     */
    public static R exec(String command) {
        return exec(command, false);
    }

    /**
     * 执行指令
     *
     * @param command 指令
     * @param isWait  是否等待
     */
    public static R exec(String command, boolean isWait) {
        R r = new R();
        try {
            logger.debug("============================= 执行指令开始 ===============================");
            logger.debug("执行指令：{}", command);
            final Process process = Runtime.getRuntime().exec(command);
            r = getProcessR(process, isWait);
        } catch (Exception e) {
            r.setExceptionStr(e.getMessage());
            logger.error("执行指令异常", e);
        }
        return r;
    }

    /**
     * 执行指令
     *
     * @param commands 指令数组
     */
    public static R exec(String[] commands) {
        return exec(commands, false);
    }

    /**
     * 执行指令
     *
     * @param commands 指令数组
     * @param isWait   是否等待
     */
    public static R exec(String[] commands, boolean isWait) {
        R r = new R();
        try {
            logger.debug("============================= 执行指令开始 ===============================");
            logger.debug("执行指令：{}", Arrays.asList(commands));
            final Process process = Runtime.getRuntime().exec(commands);
            r = getProcessR(process, isWait);
        } catch (Exception e) {
            r.setExceptionStr(e.getMessage());
            logger.error("执行指令异常", e);
        }
        return r;
    }

    /**
     * 从进城获取响应数据
     *
     * @param process 进程
     * @param isWait  是否等待
     * @return 响应
     */
    public static R getProcessR(Process process, boolean isWait) {
        R r = new R();
        try {
            if (isWait) {
                final int waitFor = process.waitFor();
                r.setWaitFor(waitFor);
            }
            r.setInputStr(getStreamStr(process.getInputStream()));
            r.setErrorStr(getStreamStr(process.getErrorStream()));
            r.setExitValue(process.exitValue());

            process.destroy();
        } catch (Exception e) {
            r.setExceptionStr(e.getMessage());
            logger.error("执行指令异常", e);
        }

        {
            logger.debug("执行指令 waitFor 代码：{}", r.getWaitFor());
            logger.debug("执行指令 exitValue 代码：{}", r.getExitValue());

            String inputStr = r.getInputStr();
            if (inputStr != null && inputStr.trim() != "" && inputStr.trim().length() > 0) {
                logger.debug("执行指令正常输出：\r\n {}", inputStr);
            }

            String errorStr = r.getErrorStr();
            if (errorStr != null && errorStr.trim() != "" && errorStr.trim().length() > 0) {
                logger.debug("执行指令警告或错误：\r\n {}", errorStr);
            }

            String exceptionStr = r.getExceptionStr();
            if (exceptionStr != null && exceptionStr.trim() != "" && exceptionStr.trim().length() > 0) {
                logger.debug("执行指令异常信息：{}", exceptionStr);
            }
            logger.debug("============================= 执行指令结束 ===============================");
        }

        return r;
    }

    /**
     * 接收的数据流取为字符串
     *
     * @param is 数据流
     * @return 字符串
     */
    public static String getStreamStr(InputStream is) {
        Charset charset = StandardCharsets.UTF_8;
        if (OS.isFamilyWindows()) {
            charset = Charset.forName("GBK");
        }
        return getStreamStr(is, charset);
    }

    /**
     * 接收的数据流取为字符串
     *
     * @param is      数据流
     * @param charset 编码
     * @return 字符串
     */
    public static String getStreamStr(InputStream is, Charset charset) {
        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader(is, charset));

            StringBuilder sb = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                sb.append(line);
                sb.append("\n");
            }

            return sb.toString();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            try {
                is.close();
                br.close();
            } catch (Exception exception) {
            }
        }
    }

    /**
     * 响应的结果
     */
    public static class R {
        /**
         * waitFor 状态码
         */
        private int waitFor;
        /**
         * 执行的正常输出信息
         */
        private String inputStr;
        /**
         * 执行过程中的警告或错误描述
         */
        private String errorStr;
        /**
         * 执行成功，默认是 0， 非 0 为不成功
         */
        private int exitValue;
        /**
         * 异常信息描述
         */
        private String exceptionStr;

        public R() {
        }

        public R(int waitFor, String inputStr, String errorStr) {
            this.waitFor = waitFor;
            this.inputStr = inputStr;
            this.errorStr = errorStr;
        }

        public R(String inputStr, String errorStr, int exitValue) {
            this.inputStr = inputStr;
            this.errorStr = errorStr;
            this.exitValue = exitValue;
        }

        public R(int waitFor, String inputStr, String errorStr, int exitValue) {
            this.waitFor = waitFor;
            this.inputStr = inputStr;
            this.errorStr = errorStr;
            this.exitValue = exitValue;
        }

        public int getWaitFor() {
            return waitFor;
        }

        public void setWaitFor(int waitFor) {
            this.waitFor = waitFor;
        }

        public String getInputStr() {
            return inputStr;
        }

        public void setInputStr(String inputStr) {
            this.inputStr = inputStr;
        }

        public String getErrorStr() {
            return errorStr;
        }

        public void setErrorStr(String errorStr) {
            this.errorStr = errorStr;
        }

        public int getExitValue() {
            return exitValue;
        }

        public void setExitValue(int exitValue) {
            this.exitValue = exitValue;
        }

        public boolean isSuccess() {
            return exitValue == 0;
        }

        public String getExceptionStr() {
            return exceptionStr;
        }

        public void setExceptionStr(String exceptionStr) {
            this.exceptionStr = exceptionStr;
        }
    }
}
