package cn.net.wanmo.common.image;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

/**
 * 缩放工具
 */
public class ImageResizeUtil {
    private static Logger logger = LoggerFactory.getLogger(ImageResizeUtil.class);

    /**
     * 图片分辨率压缩
     *
     * @param sourceFile  源图片
     * @param scaledWidth 目标宽
     * @return 结果图片
     * @throws IOException
     */
    public static File resizeWidth(File sourceFile, int scaledWidth) {
        File targetFile = null;
        try {
            targetFile = File.createTempFile("image_resize_", "." + ImageUtil.getSuffix(sourceFile));
            boolean flag = resize(sourceFile, targetFile, scaledWidth, 0);

            if (flag == false && targetFile != null) {
                targetFile.deleteOnExit();
                targetFile = null;
            }
        } catch (IOException e) {
            logger.error("图片分辨率压缩异常：" + sourceFile, e);
        }
        return targetFile;
    }

    /**
     * 图片分辨率压缩
     *
     * @param sourceFile   源图片
     * @param scaledHeight 目标高
     * @return 结果图片
     * @throws IOException
     */
    public static File resizeHeight(File sourceFile, int scaledHeight) {
        File targetFile = null;
        try {
            targetFile = File.createTempFile("image_resize_", "." + ImageUtil.getSuffix(sourceFile));
            boolean flag = resize(sourceFile, targetFile, 0, scaledHeight);

            if (flag == false && targetFile != null) {
                targetFile.deleteOnExit();
                targetFile = null;
            }
        } catch (IOException e) {
            logger.error("图片分辨率压缩异常：" + sourceFile, e);
        }
        return targetFile;
    }


    /**
     * 图片分辨率压缩
     *
     * @param sourceFile   源图片
     * @param targetFile   结果图片
     * @param scaledWidth  目标宽
     * @param scaledHeight 目标高
     * @return 目标图片
     * @throws IOException
     */
    public static boolean resize(File sourceFile, File targetFile, int scaledWidth, int scaledHeight) {
        try {
            BufferedImage bufferedImage = ImageIO.read(sourceFile);
            int width = bufferedImage.getWidth();
            int height = bufferedImage.getHeight();

            if (scaledWidth == 0 && scaledHeight == 0) {
                scaledWidth = width;
                scaledHeight = height;
            }

            if (scaledWidth == 0) {
                {
                    double percentage = scaledHeight * 1.0 / height; // 得到高度百分比
                    scaledWidth = Double.valueOf(percentage * width).intValue(); // 根据高度百分比 计算需要的宽度
                }
                {
//                double percentage = width * 1.0 / height; // 计算原图的宽高百分比
//                scaledWidth = Double.valueOf(percentage * scaledHeight).intValue(); // 根据宽高百分比 计算需要的宽度
                }
            }

            if (scaledHeight == 0) {
                {
                    double percentage = scaledWidth * 1.0 / width; // 得到宽度百分比
                    scaledHeight = Double.valueOf(percentage * height).intValue(); // 根据宽度百分比 计算需要的高度
                }
                {
//                double percentage = height * 1.0 / width; // 计算原图的高宽百分比
//                scaledHeight = Double.valueOf(percentage * scaledWidth).intValue(); // 根据高宽百分比 计算需要的高度
                }
            }

            return resize(bufferedImage, scaledWidth, scaledHeight, targetFile, ImageUtil.getFormatName(sourceFile));
        } catch (IOException e) {
            logger.error("图片分辨率压缩异常：" + sourceFile, e);
            return false;
        }
    }


    /**
     * 图片分辨率压缩
     *
     * @param sourceImage  源图片
     * @param scaledWidth  目标宽
     * @param scaledHeight 目标高
     * @param targetFile   结果图片 为空则会生成临时文件
     * @param formatName   目标格式（ jpeg / png ）
     * @return 目标文件
     * @throws IOException
     */
    public static boolean resize(BufferedImage sourceImage, int scaledWidth, int scaledHeight, File targetFile, String formatName) {
        boolean flag = false;
        try {
            // *     SSCALE_AREA_AVERAGING: 使用 Area Averaging 图像缩放算法;
            // *     SCALE_DEFAULT: 使用默认的图像缩放算法;
            // *     SCALE_SMOOTH: 选择图像平滑度比缩放速度具有更高优先级的图像缩放算法。
            // Image.SCALE_SMOOTH 的缩略算法  [生成缩略图片的平滑度的优先级比速度高  生成的图片质量比较好  但速度慢]
            Image scaledInstance = sourceImage.getScaledInstance(scaledWidth, scaledHeight, BufferedImage.SCALE_SMOOTH); // 获取压缩后的图片
            BufferedImage targetImage = new BufferedImage(scaledWidth, scaledHeight, sourceImage.getType()); // 创建待输出的图片

            {
                Graphics graphics = targetImage.getGraphics(); // 获取画笔
                graphics.drawImage(scaledInstance, 0, 0, null); // 将Image对象画在画布上,最后一个参数,ImageObserver:接收有关 Image 信息通知的异步更新接口,没用到直接传空
                graphics.dispose(); // 释放资源
            }

            ImageIO.write(targetImage, formatName, targetFile);
            flag = true;
        } catch (IOException e) {
            logger.error("图片分辨率压缩异常", e);
        }

        return flag;
    }

}
