package cn.net.wanmo.common.image;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;

/**
 * 图片文件工具类
 * https://blog.csdn.net/ycf921244819/article/details/104594670
 */
public class ImageUtil {

    private static Logger logger = LoggerFactory.getLogger(ImageUtil.class);


    /**
     * 判断文件是否为图片
     *
     * @param file 图片文件
     * @return true 是 | false 否
     */
    public static final boolean isImage(File file) {
        boolean flag = false;
        try {
            BufferedImage bufreader = ImageIO.read(file);
            if (bufreader == null) {
                flag = false;
            } else {
                int width = bufreader.getWidth();
                int height = bufreader.getHeight();

                if (width == 0 || height == 0) {
                    flag = false;
                } else {
                    flag = true;
                }
            }
        } catch (IOException e) {
            logger.error("判断文件是否为图片异常：" + file, e);
        }
        return flag;
    }

    /**
     * 获取图片文件实际类型,若不是图片则返回null
     *
     * @param file 图片文件
     * @return 文件类型
     */
    public final static String getFormatName(File file) {
        String formatName = null;
        try {
            if (isImage(file)) { // 是图片才处理
                ImageInputStream iis = ImageIO.createImageInputStream(file);
                Iterator<ImageReader> iter = ImageIO.getImageReaders(iis);
                if (!iter.hasNext()) {
                    return null;
                }

                ImageReader reader = iter.next();
                iis.close();
                formatName = reader.getFormatName().toLowerCase();
            }

        } catch (IOException e) {
            logger.error("获取图片 FormatName 异常：" + file, e);
        }

        return formatName;
    }

    /**
     * 获取图片后缀，优先获取文件名 点 后的后缀， 如果没有，则分析图片的 formatName
     *
     * @param file 图片
     * @return 后缀
     */
    public static String getSuffix(File file) {
        return getSuffix(file, true);
    }

    /**
     * 获取图片后缀，优先获取文件名 点 后的后缀， 如果没有，则分析图片的 formatName
     *
     * @param file         图片
     * @param isFormatName 是否取图片的 formatName
     * @return 后缀
     */
    public static String getSuffix(File file, boolean isFormatName) {
        String suffix = null;
        {
            String filename = getFilename(file);
            int lastIndexOf = filename.lastIndexOf(".");

            if (lastIndexOf == -1) {
                suffix = "";
            } else {
                suffix = filename.substring(lastIndexOf + 1);
            }
        }

        if (isFormatName && (suffix == null || "".equals(suffix.trim()))) {
            suffix = getFormatName(file);
        }

        return suffix;
    }


    /**
     * 没有后缀，则会追加后缀，
     * 否则会修改后缀
     *
     * @param file 图片文件
     * @return 结果文件
     */
    public static File renameSuffix(File file, String suffix) {
        if (file == null || !file.exists()) {
            logger.warn("文件不存在：{}", file);
            return file;
        }

        // 处理后缀
        if (suffix.startsWith(".")) {
            suffix = suffix.substring(1);
        }

        String absolutePath = file.getAbsolutePath();
        { // 组装新文件的路径
            String suffixTemp = getSuffix(file, false);

            if (suffixTemp == null || "".equals(suffixTemp.trim())) {
                absolutePath = absolutePath + "." + suffix;
            } else {
                int lastIndexOf = absolutePath.lastIndexOf(".");
                absolutePath = absolutePath.substring(0, lastIndexOf + 1) + suffix;
            }
        }

        File destFile = new File(absolutePath);
        file.renameTo(destFile);
        return destFile;
    }

    /**
     * 获取图片大小 KB
     *
     * @param file 图片
     * @return KB
     */
    public static Double getSizeKB(File file) {
        if (file == null || !file.exists()) {
            return 0.0;
        }

        return file.length() / 1024.0;
    }

    /**
     * 获取图片大小 MB
     *
     * @param file 图片
     * @return MB
     */
    public static Double getSizeMB(File file) {
        if (file == null || !file.exists()) {
            return 0.0;
        }

        return file.length() / 1024.0 / 1024.0;
    }


    /**
     * 获取文件名称
     *
     * @param file 文件
     * @return 文件名称
     */
    public static String getFilename(File file) {
        return file.getName();
    }
}
