package cn.net.wanmo.common.jwt;

import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.util.DateUtil;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.security.Key;

public class JwtUtil {

    /**
     * 签名盐值
     */
    public static String SIGNING = "WANMO-JWT";

    /**
     * 过期时间，单位秒
     */
    public static Integer EXPIRATION_TIME = 180;

    /**
     * 生成令牌
     *
     * @param data 数据声明
     * @return 令牌
     */
    public static String createTokenBySigning(BuilderData data) {
        return createTokenBySigning(data, SIGNING);
    }

    /**
     * 生成令牌
     *
     * @param data    数据声明
     * @param signing 签名盐值
     * @return 令牌
     */
    public static String createTokenBySigning(BuilderData data, String signing) {
        String token = Jwts.builder()
                .setId(data.getId())
                .setHeaderParams(data.getHeader())
                .setIssuer(data.getIssuer()) // 签发者
                .setIssuedAt(data.getIssuedAt()) // 签发时间
                .setSubject(data.getSubject()) // 面向的用户
                .setAudience(data.getAudience()) // 接收方
                .setNotBefore(data.getNotBefore()) // 起始时间
                .setPayload(data.getPayload()) // 负载信息
                .setExpiration(DateUtil.addSeconds(data.getExpirationTime() == null ? EXPIRATION_TIME : data.getExpirationTime())) // 截至时间
                .addClaims(data.getClaims())
                .signWith(SignatureAlgorithm.HS512, signing)
                .compact();
        return token;
    }


    /**
     * 生成令牌 <br/>
     * 注：截至时间失效
     *
     * @param data 数据声明
     * @return 令牌
     */
    public static String createTokenByKey(BuilderData data) {
        return createTokenByKey(data, generalKey());
    }

    /**
     * 生成令牌 <br/>
     * 注：截至时间失效
     *
     * @param data 数据声明
     * @param key  签名密钥
     * @return 令牌
     */
    public static String createTokenByKey(BuilderData data, Key key) {
        String token = Jwts.builder()
                .setId(data.getId())
                .setHeader(data.getHeader())
                .setIssuer(data.getIssuer()) // 签发者
                .setIssuedAt(data.getIssuedAt()) // 签发时间
                .setSubject(data.getSubject()) // 面向的用户
                .setAudience(data.getAudience()) // 接收方
                .setNotBefore(data.getNotBefore()) // 起始时间
                .setExpiration(DateUtil.addSeconds(data.getExpirationTime() == null ? EXPIRATION_TIME : data.getExpirationTime())) // 截至时间
                .setPayload(data.getPayload()) // 负载信息
                .addClaims(data.getClaims())
                .signWith(SignatureAlgorithm.HS512, key)
                .compact();
        return token;
    }

    /**
     * 从令牌中获取数据声明
     *
     * @param token 令牌
     * @return 数据声明
     */
    public static Claims parseTokenBySigning(String token) {
        return parseTokenBySigning(token, SIGNING);
    }

    /**
     * 从令牌中获取数据声明
     *
     * @param token   令牌
     * @param signing 签名盐值
     * @return 数据声明
     */
    public static Claims parseTokenBySigning(String token, String signing) {
        return Jwts.parser()
                .setSigningKey(signing)
                .parseClaimsJws(token)
                .getBody();
    }

    /**
     * 从令牌中获取数据声明
     *
     * @param token 令牌
     * @return 数据声明
     */
    public static Claims parseTokenByKey(String token) {
        return parseTokenByKey(token, generalKey());
    }

    /**
     * 从令牌中获取数据声明
     *
     * @param token 令牌
     * @param key   签名密钥
     * @return 数据声明
     */
    public static Claims parseTokenByKey(String token, Key key) {
        return Jwts.parser()
                .setSigningKey(key)
                .parseClaimsJws(token)
                .getBody();
    }

    /**
     * 由字符串生成加密key
     */
    public static SecretKey generalKey() {
        // 本地的密码解码
        byte[] encodedKey = CodecUtil.encodeBase64(SIGNING).getBytes();
        // 根据给定的字节数组使用AES加密算法构造一个密钥
        SecretKey key = new SecretKeySpec(encodedKey, 0, encodedKey.length, "AES");
        return key;
    }
}
