package cn.net.wanmo.common.mail.sun162;

import cn.net.wanmo.common.mail.sun162.exception.MailException;
import cn.net.wanmo.common.mail.sun162.pojo.MailAddress;
import cn.net.wanmo.common.mail.sun162.pojo.MailContent;

import javax.mail.Message;
import javax.mail.Multipart;
import javax.mail.Transport;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class MailSender {

    public static MailSender build(Message message) {
        MailSender sender = new MailSender();
        sender.message = message;

        return sender;
    }

    /**
     * 发送邮件
     *
     * @param subject 邮件主题
     * @param text    邮件内容
     */
    public MailSender sendText(String subject, String text) {
        try {
            this.message.setSubject(subject);
            this.message.setText(text);
        } catch (Exception e) {
            throw new MailException(e);
        }
        return this;
    }

    /**
     * 发送邮件
     *
     * @param subject 邮件主题
     * @param html    邮件内容
     */
    public MailSender sendHtml(String subject, String html) {
        try {
            this.message.setSubject(subject);
            this.message.setContent(html, "text/html; charset=utf-8");
        } catch (Exception e) {
            throw new MailException(e);
        }
        return this;
    }

    /**
     * 发送邮件
     *
     * @param subject 邮件主题
     * @param content 邮件内容
     */
    public MailSender sendContent(String subject, MailContent content) {
        try {
            Multipart multipart = new MimeMultipart();

            { // 创建邮件正文
                MimeBodyPart bodyPart = new MimeBodyPart();

                if (content.isHtml()) {
                    bodyPart.setContent(content.getText(), "text/html; charset=utf-8");
                } else {
                    bodyPart.setText(content.getText(), content.getCharset());
                }
                multipart.addBodyPart(bodyPart);
            }


            { // 创建附件部分
                if (Objects.nonNull(content.getAttach())) {
                    for (MailContent.Attach attach : content.getAttach()) {
                        MimeBodyPart attachmentPart = new MimeBodyPart();
                        {
                            String attachId = attach.getAttachId();
                            if (Objects.nonNull(attachId) && attachId.trim().length() > 0) {
                                attachmentPart.setHeader("Content-ID", attachId);
                            }
                        }
                        attachmentPart.attachFile(attach.getFilePath());
                        {
                            String fileName = attach.getFileName();
                            if (Objects.nonNull(fileName) && fileName.trim().length() > 0) {
                                attachmentPart.setFileName(MimeUtility.encodeText(fileName));
                            }
                        }

                        multipart.addBodyPart(attachmentPart);
                    }
                }
            }

            // 设置邮件内容
            message.setSubject(subject);
            message.setContent(multipart);
        } catch (Exception e) {
            throw new MailException(e);
        }
        return this;
    }

    public void ok() {
        try {
            if (Objects.nonNull(this.toMail)) { // 邮件接收者
                message.setRecipient(Message.RecipientType.TO, this.toMail.toInternetAddress());
            }
            if (Objects.nonNull(this.toMails)) { // 邮件群接受者
                for (MailAddress mailAddress : this.toMails) {
                    message.addRecipient(Message.RecipientType.TO, mailAddress.toInternetAddress());
                }
            }
            if (Objects.nonNull(this.ccMails)) { // 邮件抄送者 【垃圾邮件解决问题：抄送人添加自己】
                for (MailAddress mailAddress : this.ccMails) {
                    message.addRecipient(Message.RecipientType.CC, mailAddress.toInternetAddress());
                }
            }
            if (Objects.nonNull(this.bccMails)) { // 邮件密送者
                for (MailAddress mailAddress : this.bccMails) {
                    message.addRecipient(Message.RecipientType.BCC, mailAddress.toInternetAddress());
                }
            }

            Transport.send(message); // 发送邮件
        } catch (Exception e) {
            throw new MailException(e);
        }
    }


    /**
     * 字符编码
     */
    private String charset = "UTF-8";
    /**
     * 消息对象
     */
    private Message message;

    /**
     * 邮件接收者
     */
    private MailAddress toMail;

    /**
     * 邮件接收者
     */
    private List<MailAddress> toMails = new ArrayList<>();

    /**
     * 邮件抄送者
     */
    private List<MailAddress> ccMails = new ArrayList<>();

    /**
     * 邮件密送者
     */
    private List<MailAddress> bccMails = new ArrayList<>();

    public String getCharset() {
        return charset;
    }

    public MailSender setCharset(String charset) {
        this.charset = charset;
        return this;
    }

    public Message getMessage() {
        return message;
    }

    public MailSender setMessage(Message message) {
        this.message = message;
        return this;
    }

    public MailAddress getToMail() {
        return toMail;
    }

    public MailSender setToMail(MailAddress toMail) {
        this.toMail = toMail;
        return this;
    }

    public List<MailAddress> getToMails() {
        return toMails;
    }

    public MailSender setToMails(List<MailAddress> toMails) {
        this.toMails = toMails;
        return this;
    }

    public List<MailAddress> getCcMails() {
        return ccMails;
    }

    public MailSender setCcMails(List<MailAddress> ccMails) {
        this.ccMails = ccMails;
        return this;
    }

    public List<MailAddress> getBccMails() {
        return bccMails;
    }

    public MailSender setBccMails(List<MailAddress> bccMails) {
        this.bccMails = bccMails;
        return this;
    }
}
