package cn.net.wanmo.common.minio;

import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.StringUtil;
import io.minio.*;
import io.minio.http.Method;
import io.minio.messages.DeleteError;
import io.minio.messages.DeleteObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class ObjectUtil {
    private static Logger logger = LoggerFactory.getLogger(ObjectUtil.class);


    /**
     * 文件上传
     *
     * @param bucketName    操作的桶名称
     * @param saveFilename  上传以后的文件名
     * @param localFilepath 上传的本地文件
     */
    public static void uploadObject(String bucketName, String saveFilename, String localFilepath) {
        try {
            BucketUtil.makeBucket(bucketName);

            MinioConfig.client.uploadObject(
                    UploadObjectArgs.builder()
                            .bucket(bucketName)
                            .object(saveFilename)
                            .filename(localFilepath)
                            .build());

            logger.debug("'{}' is successfully uploaded as object '{}' to bucket '{}'.", localFilepath, saveFilename, bucketName);
        } catch (Exception e) {
            logger.error("文件上传 MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 文件上传
     *
     * @param bucketName   操作的桶名称
     * @param saveFilename 上传以后的文件名
     * @param is           待上传的文件流
     */
    public static void putObject(String bucketName, String saveFilename, InputStream is) {
        putObject(bucketName, saveFilename, is, null);
    }

    /**
     * 文件上传
     *
     * @param bucketName   操作的桶名称
     * @param saveFilename 上传以后的文件名
     * @param is           待上传的文件流
     * @param contentType           待上传的文件类型
     */
    public static void putObject(String bucketName, String saveFilename, InputStream is, String contentType) {
        try {
            BucketUtil.makeBucket(bucketName);

            MinioConfig.client.putObject(PutObjectArgs.builder()
                    .contentType(StringUtil.isBlank(contentType) ? null : contentType)
                    .bucket(bucketName)
                    .object(saveFilename)
                    .stream(is, is.available(), -1)
                    .build());
        } catch (Exception e) {
            logger.error("文件上传 MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
    }


    /**
     * 已流的形式获取文件对象
     *
     * @param bucketName       操作的桶名称
     * @param downloadFilename 下载的文件名称
     */
    public static InputStream getObject(String bucketName, String downloadFilename) {
        InputStream stream;
        try {
            stream = MinioConfig.client.getObject(GetObjectArgs.builder()
                    .bucket(bucketName)
                    .object(downloadFilename)
                    .build());
        } catch (Exception e) {
            logger.error("文件下载 MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
        return stream;
    }

    /**
     * 已流的形式获取文件对象（断点下载）
     *
     * @param bucketName       操作的桶名称
     * @param downloadFilename 下载的文件名称
     * @param offset       起始字节的位置
     * @param length 要读取的长度（可选，如果无值，则代表到文件结尾）
     */
    public static InputStream getObject(String bucketName, String downloadFilename, Long offset, Long length) {
        InputStream stream;
        try {
            stream = MinioConfig.client.getObject(GetObjectArgs.builder()
                    .bucket(bucketName)
                    .object(downloadFilename)
                    .offset(offset)
                    .length(length)
                    .build());
        } catch (Exception e) {
            logger.error("文件下载 MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
        return stream;
    }

    /**
     * 文件下载，并保存到本地
     *
     * @param bucketName       操作的桶名称
     * @param downloadFilename 下载的文件名称
     * @param localFilepath    下载本地保存路径
     */
    public static void downloadObject(String bucketName, String downloadFilename, String localFilepath) {
        try {
            MinioConfig.client.downloadObject(DownloadObjectArgs.builder().bucket(bucketName).object(downloadFilename).filename(localFilepath).build());
            logger.debug("'{}' is successfully downloaded as object '{}' from bucket '{}'.", localFilepath, downloadFilename, bucketName);
        } catch (Exception e) {
            logger.error("文件下载 MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 文件移除
     *
     * @param bucketName     操作的桶名称
     * @param removeFilename 移除的文件名称
     */
    public static void removeObject(String bucketName, String removeFilename) {
        try {
            MinioConfig.client.removeObject(RemoveObjectArgs.builder().bucket(bucketName).object(removeFilename).build());
            logger.debug("successfully removed as object '{}' from bucket '{}'.", removeFilename, bucketName);
        } catch (Exception e) {
            logger.error("文件移除 MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 文件移除指定桶的多个文件
     *
     * @param bucketName     操作的桶名称
     * @param removeFilenames 多个移除的文件名称
     */
    public static void removeObjects(String bucketName, List<String> removeFilenames) {
        try {
            List<DeleteObject> objects = new LinkedList<>();
            for (String objectName : removeFilenames){
                objects.add(new DeleteObject(objectName));
            }

            final Iterable<Result<DeleteError>> results = MinioConfig.client.removeObjects(RemoveObjectsArgs.builder().bucket(bucketName).objects(objects).build());

            for (Result<DeleteError> result : results) {
                final DeleteError error = result.get();
                logger.error("Error in deleting object {} , errMsg {}", error.objectName(), error.message());
            }

            logger.debug("successfully removed as objects '{}' from bucket '{}'.", removeFilenames, bucketName);
        } catch (Exception e) {
            logger.error("文件移除 MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 生成预签名URL
     * @param bucketName 桶名称
     * @param fileName 文件路径及名称
     * @param contentType 预览类型 image/gif", "image/jpeg", "image/jpg", "image/png", "application/pdf
     */
    public static String presignedGetObjectUrl(String bucketName, String fileName, String contentType) {
        try {
            Map<String, String> queryParams = new HashMap<>();
            queryParams.put("response-content-type", contentType == null ? "application/pdf" : contentType);

            return MinioConfig.client.getPresignedObjectUrl(GetPresignedObjectUrlArgs.builder()
                    .method(Method.GET)
                    .bucket(bucketName)
                    .object(fileName)
                    .expiry(7, TimeUnit.DAYS)
//                    .expiry(60 * 60 * 24) // 生成的预签名 URL 可访问的有效时间，最大期限7天
                    .extraQueryParams(queryParams)
                    .build());
        } catch (Exception e) {
            logger.error("生成预签名URL MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 生成预签名URL
     * @param bucketName 桶名称
     * @param fileName 文件路径及名称
     * @param contentType 预览类型 image/gif", "image/jpeg", "image/jpg", "image/png", "application/pdf
     */
    public static String presignedPostObjectUrl(String bucketName, String fileName, String contentType) {
        try {
            Map<String, String> queryParams = new HashMap<>();
            queryParams.put("response-content-type", contentType == null ? "application/pdf" : contentType);

            return MinioConfig.client.getPresignedObjectUrl(GetPresignedObjectUrlArgs.builder()
                    .method(Method.POST)
                    .bucket(bucketName)
                    .object(fileName)
                    .expiry(7, TimeUnit.DAYS)
//                    .expiry(60 * 60 * 24) // 生成的预签名 URL 可访问的有效时间，最大期限7天
                    .extraQueryParams(queryParams)
                    .build());
        } catch (Exception e) {
            logger.error("生成预签名URL MinIO 发生异常", e);
            throw Exceptions.unchecked(e);
        }
    }
}
