package cn.net.wanmo.quartz.job;

import org.quartz.*;
import org.quartz.impl.StdSchedulerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.List;

@DisallowConcurrentExecution
@PersistJobDataAfterExecution
public abstract class BaseJob<CustomJob extends Job> {
    private final Logger logger = LoggerFactory.getLogger(BaseJob.class);

    private Class<CustomJob> customJobClass;

    protected void setCustomJobClass(Class<CustomJob> customJobClass) {
        this.customJobClass = customJobClass;
    }

    /**
     * 创建 JobDetail
     * @param name 名称
     * @param group 所属组
     * @return JobDetail
     */
    public JobDetail buildJobDetail(String name, String group) {
        return buildJobDetail(name, group, null, null);
    }

    /**
     * 创建 JobDetail <br />
     * @param name 名称
     * @param group 所属组
     * @param jobDataMap 携带数据
     * @return JobDetail
     */
    public JobDetail buildJobDetail(String name, String group, JobDataMap jobDataMap) {
        return buildJobDetail(name, group, jobDataMap, null);
    }

    /**
     * 创建 JobDetail
     * @param name 名称
     * @param group 所属组
     * @param jobDescription 描述
     * @return JobDetail
     */
    public JobDetail buildJobDetail(String name, String group, String jobDescription) {
        return buildJobDetail(name, group, null, jobDescription);
    }

    /**
     * 创建 JobDetail <br />
     * @param name 名称
     * @param group 所属组
     * @param jobDataMap 携带数据
     * @param jobDescription 描述
     * @return JobDetail
     */
    public JobDetail buildJobDetail(String name, String group, JobDataMap jobDataMap, String jobDescription) {
        jobDataMap = jobDataMap == null ? new JobDataMap() : jobDataMap;

        return JobBuilder.newJob(customJobClass)
                .withIdentity(name, group)
                .usingJobData(jobDataMap)
                .withDescription(jobDescription)
                .build();
    }

    /**
     * 创建 Trigger
     * @param name 名称
     * @param group 所属组
     * @param scheduleBuilder 执行计划
     * @return Trigger
     */
    public Trigger buildTrigger(String name, String group, ScheduleBuilder scheduleBuilder) {
        return buildTrigger(name, group, scheduleBuilder, null, null, null, 5, null);
    }

    /**
     * 创建 Trigger
     * @param name 名称
     * @param group 所属组
     * @param scheduleBuilder 执行计划
     * @param jobDataMap 携带数据
     * @return Trigger
     */
    public Trigger buildTrigger(String name, String group, ScheduleBuilder scheduleBuilder, JobDataMap jobDataMap) {
        return buildTrigger(name, group, scheduleBuilder, jobDataMap, null, null, 5, null);
    }

    /**
     * 创建 Trigger <br />
     * @param name 名称
     * @param group 所属组
     * @param scheduleBuilder 执行计划
     * @param jobDataMap 携带数据
     * @param triggerStartTime 开始时间
     * @param triggerEndTime 结束时间
     * @param triggerPriority 优先级
     * @param triggerDescription 描述
     * @return Trigger
     */
    public Trigger buildTrigger(String name, String group, ScheduleBuilder scheduleBuilder, JobDataMap jobDataMap, Date triggerStartTime, Date triggerEndTime, int triggerPriority, String triggerDescription) {
        jobDataMap = jobDataMap == null ? new JobDataMap() : jobDataMap;

        return TriggerBuilder.newTrigger()
                .withIdentity(name, group)
                .usingJobData(jobDataMap)
                .withDescription(triggerDescription)
                .startAt(triggerStartTime)
                .endAt(triggerEndTime)
                .withPriority(triggerPriority)
                .startNow()
                .withSchedule(scheduleBuilder)
                .build();
    }

    /**
     * 启动任务
     * @param jobDetail 任务明细
     * @param trigger 任务触发器
     * @return Scheduler
     */
    public Scheduler start(JobDetail jobDetail, Trigger trigger) {
        try {
            Scheduler scheduler = StdSchedulerFactory.getDefaultScheduler();

            { // 判断任务是否已经存在
                JobKey jobKey = jobDetail.getKey();

                if (scheduler.checkExists(jobKey)) {
                    logger.debug("任务已存在：" + jobKey);
                    scheduler.resumeJob(jobKey);
                } else {
                    logger.debug("任务不存在：" + jobKey);
                    scheduler.scheduleJob(jobDetail, trigger);
                }

                logger.debug("任务启动：" + jobKey);
            }

            scheduler.start();
            return scheduler;
        } catch (SchedulerException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 启动任务
     * @param jobName 任务名称
     * @param triggerName 触发器名称
     * @param groupName 分组名称
     * @param cronExpression cron 表达式
     * @return Scheduler
     */
    public Scheduler start(String jobName, String triggerName, String groupName, String cronExpression) {
        JobDetail jobDetail = buildJobDetail(jobName, groupName);

        CronScheduleBuilder cronScheduleBuilder = CronScheduleBuilder.cronSchedule(cronExpression);
        Trigger trigger = buildTrigger(triggerName, groupName, cronScheduleBuilder);

        return start(jobDetail, trigger);
    }
}
