package cn.net.wanmo.common.security;

import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.util.Exceptions;
import org.apache.commons.lang3.Validate;

import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.SecureRandom;

/**
 * 支持SHA-1/MD5消息摘要的工具类.<br>
 * 返回ByteSource，可进一步被编码为Hex, Base64或UrlSafeBase64
 */
public class Digests {

	public static final String ALGORITHM_MD5 = "MD5";
	public static final String ALGORITHM_SHA1 = "SHA-1";
	public static final String ALGORITHM_SHA256 = "SHA-256";
	public static final String ALGORITHM_SHA512 = "SHA-512";
	public static SecureRandom random = new SecureRandom();

	/****************************************** md5 start ******************************************************/

	/**
	 * 对输入字符串进行md5散列.
	 *
	 * @param data 字符串数据
	 * @return 摘要
	 */
	public static String md5Hex(byte[] data) {
		return md5Hex(data, null, 1);
	}

	/**
	 * 对输入字符串进行md5散列.
	 *
	 * @param data      字符串数据
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String md5Hex(byte[] data, int iterations) {
		return md5Hex(data, null, iterations);
	}

	/**
	 * 对输入字符串进行md5散列.
	 *
	 * @param data      字符串数据
	 * @param salt       盐值
	 * @return 摘要
	 */
	public static String md5Hex(byte[] data, byte[] salt) {
		return md5Hex(data, salt, 1);
	}

	/**
	 * 对输入字符串进行md5散列.
	 *
	 * @param data      字符串数据
	 * @param salt       盐值
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String md5Hex(byte[] data, byte[] salt, int iterations) {
		return digestHex(ALGORITHM_MD5, data, salt, iterations);
	}

	/**
	 * 对文件进行md5散列.
	 *
	 * @param is 文件数据流
	 * @return 摘要
	 */
	public static String md5Hex(InputStream is) {
		return digestHex(ALGORITHM_MD5, is);
	}

	/****************************************** md5 end ******************************************************/

	/****************************************** sha1 start ******************************************************/

	/**
	 * 对输入字符串进行 sha1 散列.
	 *
	 * @param data 字符串数据
	 * @return 摘要
	 */
	public static String sha1Hex(byte[] data) {
		return sha1Hex(data, null, 1);
	}

	/**
	 * 对输入字符串进行sha1散列.
	 *
	 * @param data 字符串数据
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String sha1Hex(byte[] data, int iterations) {
		return sha1Hex(data, null, iterations);
	}

	/**
	 * 对输入字符串进行sha1散列.
	 *
	 * @param data 字符串数据
	 * @param salt  盐值
	 * @return 摘要
	 */
	public static String sha1Hex(byte[] data, byte[] salt) {
		return sha1Hex(data, salt, 1);
	}

	/**
	 * 对输入字符串进行sha1散列.
	 *
	 * @param data      字符串数据
	 * @param salt       盐值
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String sha1Hex(byte[] data, byte[] salt, int iterations) {
		return digestHex(ALGORITHM_SHA1, data, salt, iterations);
	}

	/**
	 * 对文件进行sha1散列.
	 *
	 * @param is 文件数据
	 * @return 摘要
	 */
	public static String sha1Hex(InputStream is) {
		return digestHex(ALGORITHM_SHA1, is);
	}
	/****************************************** sha1 end ******************************************************/



	/****************************************** sha256 start ******************************************************/

	/**
	 * 对输入字符串进行 sha256 散列.
	 *
	 * @param data 字符串数据
	 * @return 摘要
	 */
	public static String sha256Hex(byte[] data) {
		return sha256Hex(data, null, 1);
	}

	/**
	 * 对输入字符串进行 sha256 散列.
	 *
	 * @param data 字符串数据
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String sha256Hex(byte[] data, int iterations) {
		return sha256Hex(data, null, iterations);
	}

	/**
	 * 对输入字符串进行 sha256 散列.
	 *
	 * @param data 字符串数据
	 * @param salt  盐值
	 * @return 摘要
	 */
	public static String sha256Hex(byte[] data, byte[] salt) {
		return sha256Hex(data, salt, 1);
	}

	/**
	 * 对输入字符串进行 sha256 散列.
	 *
	 * @param data      字符串数据
	 * @param salt       盐值
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String sha256Hex(byte[] data, byte[] salt, int iterations) {
		return digestHex(ALGORITHM_SHA256, data, salt, iterations);
	}

	/**
	 * 对文件进行 sha256 散列.
	 *
	 * @param is 文件数据
	 * @return 摘要
	 */
	public static String sha256Hex(InputStream is) {
		return digestHex(ALGORITHM_SHA256, is);
	}
	/****************************************** sha256 end ******************************************************/


	/****************************************** sha512 start ******************************************************/

	/**
	 * 对输入字符串进行 sha512 散列.
	 *
	 * @param data 字符串数据
	 * @return 摘要
	 */
	public static String sha512Hex(byte[] data) {
		return sha512Hex(data, null, 1);
	}

	/**
	 * 对输入字符串进行 sha512 散列.
	 *
	 * @param data 字符串数据
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String sha512Hex(byte[] data, int iterations) {
		return sha512Hex(data, null, iterations);
	}

	/**
	 * 对输入字符串进行 sha512 散列.
	 *
	 * @param data 字符串数据
	 * @param salt  盐值
	 * @return 摘要
	 */
	public static String sha512Hex(byte[] data, byte[] salt) {
		return sha512Hex(data, salt, 1);
	}

	/**
	 * 对输入字符串进行 sha512 散列.
	 *
	 * @param data      字符串数据
	 * @param salt       盐值
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String sha512Hex(byte[] data, byte[] salt, int iterations) {
		return digestHex(ALGORITHM_SHA512, data, salt, iterations);
	}

	/**
	 * 对文件进行 sha512 散列.
	 *
	 * @param is 文件数据
	 * @return 摘要
	 */
	public static String sha512Hex(InputStream is) {
		return digestHex(ALGORITHM_SHA512, is);
	}
	/****************************************** sha512 end ******************************************************/

	/**
	 * 对字符串进行散列, 支持 md5、sha1、sha256、sha512 算法.
	 *
	 * @param algorithm  摘要算法
	 * @param input      需要摘要的数据
	 * @return 字节数组
	 */
	public static byte[] digest(String algorithm, byte[] input) {
		return digest(algorithm, input, null, 1);
	}

	/**
	 * 对字符串进行散列, 支持 md5、sha1、sha256、sha512 算法.
	 *
	 * @param algorithm  摘要算法
	 * @param input      需要摘要的数据
	 * @param iterations 迭代次数
	 * @return 字节数组
	 */
	public static byte[] digest(String algorithm, byte[] input, int iterations) {
		return digest(algorithm, input, null, iterations);
	}

	/**
	 * 对字符串进行散列, 支持 md5、sha1、sha256、sha512 算法.
	 *
	 * @param algorithm  摘要算法
	 * @param input      需要摘要的数据
	 * @param salt       盐值
	 * @return 字节数组
	 */
	public static byte[] digest(String algorithm, byte[] input, byte[] salt) {
		return digest(algorithm, input, salt, 1);
	}


	/**
	 * 对字符串进行散列, 支持 md5、sha1、sha256、sha512 算法.
	 *
	 * @param algorithm  摘要算法
	 * @param input      需要摘要的数据
	 * @param salt       盐值
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static byte[] digest(String algorithm, byte[] input, byte[] salt, int iterations) {
		try {
			MessageDigest messageDigest = MessageDigest.getInstance(algorithm);

			if (salt != null) {
				messageDigest.update(salt);
			}

			byte[] result = messageDigest.digest(input);

			for (int i = 1; i < iterations; i++) {
				messageDigest.reset();
				result = messageDigest.digest(result);
			}
			return result;
		} catch (GeneralSecurityException e) {
			throw Exceptions.unchecked(e);
		}
	}


	/**
	 * 对字符串进行散列, 支持 md5、sha1、sha256、sha512 算法.
	 *
	 * @param algorithm  摘要算法
	 * @param input      需要摘要的数据
	 * @param salt       盐值
	 * @param iterations 迭代次数
	 * @return 摘要
	 */
	public static String digestHex(String algorithm, byte[] input, byte[] salt, int iterations) {
		final byte[] digest = digest(algorithm, input, salt, iterations);
		return CodecUtil.encodeHex(digest);
	}

	/**
	 * 对文件进行散列, 支持 md5、sha1、sha256、sha512 算法.
	 *
	 * @param algorithm 摘要算法
	 * @param is     需要摘要的数据
	 * @return 摘要
	 */
	public static byte[] digest(String algorithm, InputStream is) {
		try {
			MessageDigest messageDigest = MessageDigest.getInstance(algorithm);
			int bufferLength = 8 * 1024;
			byte[] buffer = new byte[bufferLength];
			int read = is.read(buffer, 0, bufferLength);

			while (read > -1) {
				messageDigest.update(buffer, 0, read);
				read = is.read(buffer, 0, bufferLength);
			}

			return messageDigest.digest();
		} catch (Exception e) {
			throw Exceptions.unchecked(e);
		}
	}


	/**
	 * 对文件进行散列, 支持 md5、sha1、sha256、sha512 算法.
	 *
	 * @param algorithm 摘要算法
	 * @param is     需要摘要的数据
	 * @return 摘要
	 */
	public static String digestHex(String algorithm, InputStream is) {
		final byte[] digest = digest(algorithm, is);
		return CodecUtil.encodeHex(digest);
	}

	/**
	 * 生成随机的 byte[] 作为salt.
	 *
	 * @param numBytes byte 数组的大小
	 */
	public static byte[] generateSalt(int numBytes) {
		Validate.isTrue(numBytes > 0, "numBytes argument must be a positive integer (1 or larger)", numBytes);

		byte[] bytes = new byte[numBytes];
		random.nextBytes(bytes);
		return bytes;
	}

}
