package cn.net.wanmo.common.codec;

import cn.net.wanmo.common.charset.CharsetUtil;
import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.StringUtil;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringEscapeUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;

/**
 * 编码解码工具类<br/>
 * 说明：
 *   new sun.misc.BASE64Encoder().encode(bytes)将在未来版本中删除，<br/>
 *   由 java.util.Base64.getEncoder().encodeToString(bytes) 替代
 */
public class CodecUtil {

	/** 默认 CharSet UTF8 */
	public static final String UTF8_NAME = CharsetUtil.UTF8_NAME;
	/** 默认 CharSet UTF8 */
	public static final Charset UTF8 = CharsetUtil.UTF8;


	/**
	 * 将签名值转为 16进制，并且缺位补 0
	 * 效果同 encodeHex()
	 *
	 * @param digest 签名值
	 * @return 补位后的 16进制签名值
	 */
	public static String toHex(byte[] digest) {
		StringBuilder sb = new StringBuilder();

		for (byte b : digest) {
			// 把密文转换成 16 进制
			String s = Integer.toHexString(b & 0xff);
			// 判断如果密文的长度是 1， 需要在高位进行补 0
			if (s.length() == 1) {
				s = "0" + s;
			}
			sb.append(s);
		}

		return sb.toString();
	}

	/**
	 * Hex编码. 将数据转为 16进制，并且缺位补 0
	 */
	public static String encodeHex(byte[] input) {
		return new String(Hex.encodeHex(input));
	}

	/**
	 * Hex解码.
	 */
	public static byte[] decodeHex(String input) {
		try {
			if (input == null) {
				return null;
			}

			return Hex.decodeHex(input.toCharArray());
		} catch (DecoderException e) {
			throw Exceptions.unchecked(e);
		}
	}

	/**
	 * Base64编码.
	 */
	public static String encodeBase64(byte[] bytes) {
		return encodeBase64(bytes, UTF8);
	}

	/**
	 * Base64编码.
	 */
	public static String encodeBase64(byte[] bytes, Charset charset) {
		byte[] encodeBase64 = Base64.encodeBase64(bytes);
		return StringUtil.toEncodedString(encodeBase64, charset);
	}


	/**
	 * Base64编码.
	 */
	public static String encodeBase64(String input) {
		return encodeBase64(input, UTF8);
	}

	/**
	 * Base64编码.
	 */
	public static String encodeBase64(String input, Charset charset) {
		if (input == null) {
			return null;
		}

		byte[] bytes = input.getBytes(charset);
		return encodeBase64(bytes, charset);
	}

	/**
	 * Base64解码.
	 */
	public static byte[] decodeBase64(String input) {
		return decodeBase64(input, UTF8);
	}

	/**
	 * Base64解码.
	 */
	public static byte[] decodeBase64(String input, Charset charset) {
		if (input == null) {
			return null;
		}

		byte[] bytes = input.getBytes(charset);
		return Base64.decodeBase64(bytes);
	}

	/**
	 * Base64解码.
	 */
	public static String decodeBase64String(String input) {
		return decodeBase64String(input, UTF8);
	}

	/**
	 * Base64解码.
	 */
	public static String decodeBase64String(String input, Charset charset) {
		byte[] decodeBase64 = decodeBase64(input, charset);
		return StringUtil.toEncodedString(decodeBase64, charset);
	}

	/**
	 * HTML 转码.
	 */
	public static String escapeHtml(String html) {
		return StringEscapeUtils.escapeHtml4(html);
	}

	/**
	 * HTML 解码.
	 */
	public static String unescapeHtml(String htmlEscaped) {
		return StringEscapeUtils.unescapeHtml4(htmlEscaped);
	}

	/**
	 * XML 转码.
	 */
	public static String escapeXml(String xml) {
		return StringEscapeUtils.escapeXml10(xml);
	}

	/**
	 * XML 解码.
	 */
	public static String unescapeXml(String xmlEscaped) {
		return StringEscapeUtils.unescapeXml(xmlEscaped);
	}

	/**
	 * URL 编码, Encode默认为UTF-8.
	 */
	public static String urlEncode(String part) {
		return urlEncode(part, UTF8_NAME);
	}

	/**
	 * URL 编码, Encode默认为UTF-8.
	 */
	public static String urlEncode(String part, String charsetName) {
		try {
			return URLEncoder.encode(part, charsetName);
		} catch (UnsupportedEncodingException e) {
			throw Exceptions.unchecked(e);
		}
	}

	/**
	 * URL 解码, Encode默认为UTF-8.
	 */
	public static String urlDecode(String part) {
		return urlDecode(part, UTF8_NAME);
	}

	/**
	 * URL 解码, Encode默认为UTF-8.
	 */
	public static String urlDecode(String part, String charsetName) {

		try {
			return URLDecoder.decode(part, charsetName);
		} catch (UnsupportedEncodingException e) {
			throw Exceptions.unchecked(e);
		}
	}

}
