package cn.net.wanmo.common.util;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Locale;
import java.util.Set;

/**
 * Java8 后的时间工具类
 */
public class DateTimeUtil {

    /**
     * 当前的 Date
     *
     * @return Date
     */
    public static Date nowDate() {
        return toDate(nowLocalDateTime());
    }


    /**
     * 当前的 LocalDate
     *
     * @return LocalDate
     */
    public static LocalDate nowLocalDate() {
        return LocalDate.now(getZoneIdDefault());
    }

    /**
     * 当前的 LocalTime
     *
     * @return LocalTime
     */
    public static LocalTime nowLocalTime() {
        return LocalTime.now(getZoneIdDefault());
    }

    /**
     * 当前的 LocalDateTime
     *
     * @return LocalDateTime
     */
    public static LocalDateTime nowLocalDateTime() {
        return LocalDateTime.now(getZoneIdDefault());
    }


    /**
     * 当前的 LocalDateTime "yyyy-MM-dd"
     *
     * @return LocalDateTime
     */
    public static String nowDay() {
        return formatDay(nowLocalDate());
    }

    /**
     * 当前的 LocalDateTime "yyyy-MM-dd"
     *
     * @return LocalDateTime
     */
    public static String nowTime() {
        return formatTime(nowLocalTime());
    }

    /**
     * 当前的 LocalDateTime "yyyy-MM-dd HH:mm:ss"
     *
     * @return LocalDateTime
     */
    public static String nowDayTime() {
        return formatDayTime(nowLocalDateTime());
    }


    /**
     * 格式化 "HH:mm:ss"
     *
     * @param localTime localTime
     * @return 日期
     */
    public static String formatTime(LocalTime localTime) {
        return formatTime(localTime, "HH:mm:ss");
    }

    /**
     * 格式化 "HH:mm:ss"
     *
     * @param localTime localTime
     * @param pattern   "HH:mm:ss"
     * @return 日期
     */
    public static String formatTime(LocalTime localTime, String pattern) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern, Locale.CHINA);
        return localTime.format(formatter);
    }

    /**
     * 格式化 "HH:mm:ss"
     *
     * @param localDateTime localDateTime
     * @return 日期
     */
    public static String formatTime(LocalDateTime localDateTime) {
        LocalTime localTime = localDateTime.toLocalTime();
        return formatTime(localTime);
    }


    /**
     * 格式化 "HH:mm:ss"
     *
     * @param date date
     * @return 日期
     */
    public static String formatTime(Date date) {
        LocalTime localTime = toLocalTime(date);
        return formatTime(localTime);
    }


    /**
     * 格式化 "yyyy-MM-dd"
     *
     * @param localDate localDate
     * @return 日期
     */
    public static String formatDay(LocalDate localDate) {
        return formatDay(localDate, "yyyy-MM-dd");
    }

    /**
     * 格式化 "yyyy-MM-dd"
     *
     * @param localDate localDate
     * @param pattern   "yyyy-MM-dd"
     * @return 日期
     */
    public static String formatDay(LocalDate localDate, String pattern) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern, Locale.CHINA);
        return localDate.format(formatter);
    }

    /**
     * 格式化 "yyyy-MM-dd"
     *
     * @param localDateTime localDateTime
     * @return 日期
     */
    public static String formatDay(LocalDateTime localDateTime) {
        LocalDate localDate = localDateTime.toLocalDate();
        return formatDay(localDate);
    }


    /**
     * 格式化 "yyyy-MM-dd"
     *
     * @param date date
     * @return 日期
     */
    public static String formatDay(Date date) {
        LocalDate localDate = toLocalDate(date);
        return formatDay(localDate);
    }


    /**
     * 格式化 "yyyy-MM-dd HH:mm:ss"
     *
     * @param localDateTime localDateTime
     * @return 日期
     */
    public static String formatDayTime(LocalDateTime localDateTime) {
        return formatDayTime(localDateTime, "yyyy-MM-dd HH:mm:ss");
    }

    /**
     * 格式化 "yyyy-MM-dd HH:mm:ss"
     *
     * @param localDateTime localDateTime
     * @param pattern       "yyyy-MM-dd HH:mm:ss"
     * @return 日期
     */
    public static String formatDayTime(LocalDateTime localDateTime, String pattern) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern, Locale.CHINA);
        return localDateTime.format(formatter);
    }

    /**
     * 格式化 "yyyy-MM-dd HH:mm:ss"
     *
     * @param date date
     * @return 日期
     */
    public static String formatDayTime(Date date) {
        LocalDateTime localDateTime = toLocalDateTime(date);
        return formatDayTime(localDateTime);
    }


    /**
     * 格式化 "yyyy-MM-dd HH:mm:ss"
     *
     * @param date    date
     * @param pattern "yyyy-MM-dd HH:mm:ss"
     * @return 日期
     */
    public static String formatDate(Date date, String pattern) {
        LocalDateTime localDateTime = toLocalDateTime(date);
        return formatDayTime(localDateTime, pattern);
    }


    /**
     * 解析日期
     *
     * @param date 日期: such as "2007-12-03", not null
     * @return LocalDate
     */
    public static LocalDate parseLocalDate(String date) {
        return LocalDate.parse(date);
    }

    /**
     * 解析日期
     *
     * @param date    日期: such as "2007-12-03", not null
     * @param pattern 格式：such as 'yyyy-MM-dd'.
     * @return LocalDateTime
     */
    public static LocalDate parseLocalDate(String date, String pattern) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern, Locale.CHINA);
        return LocalDate.parse(date, formatter);
    }


    /**
     * 解析时间
     *
     * @param time 时间: such as "10:15:30", not null
     * @return LocalTime
     */
    public static LocalTime parseLocalTime(String time) {
        return LocalTime.parse(time);
    }

    /**
     * 解析时间
     *
     * @param time    时间: such as "10:15:30", not null
     * @param pattern 格式：such as 'HH:mm:ss'.
     * @return LocalTime
     */
    public static LocalTime parseLocalTime(String time, String pattern) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern, Locale.CHINA);
        return LocalTime.parse(time, formatter);
    }

    /**
     * 解析日期
     *
     * @param date 日期: such as '2011-12-03T10:15:30'.
     * @return LocalDateTime
     */
    public static LocalDateTime parseLocalDateTime(String date) {
        return LocalDateTime.parse(date);
    }

    /**
     * 解析日期
     *
     * @param date    日期: such as '2011-12-03 10:15:30'.
     * @param pattern 格式：such as 'yyyy-MM-dd HH:mm:ss'.
     * @return LocalDateTime
     */
    public static LocalDateTime parseLocalDateTime(String date, String pattern) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern, Locale.CHINA);
        return LocalDateTime.parse(date, formatter);
    }

    /**
     * 解析日期
     *
     * @param date 日期: such as '2011-12-03T10:15:30'.
     * @return Date
     */
    public static Date parseDate(String date) {
        LocalDateTime localDateTime = parseLocalDateTime(date);
        return toDate(localDateTime);
    }


    /**
     * 解析日期
     *
     * @param date    日期: such as '2011-12-03 10:15:30'.
     * @param pattern 格式：such as 'yyyy-MM-dd HH:mm:ss'.
     * @return Date
     */
    public static Date parseDate(String date, String pattern) {
        LocalDateTime localDateTime = parseLocalDateTime(date, pattern);
        return toDate(localDateTime);
    }


    /**
     * 转 Date，如： 2024-10-13 00:00:00
     *
     * @param localDate localDate
     * @return Date
     */
    public static Date toDate(LocalDate localDate) {
        return toDate(localDate, null);
    }

    /**
     * 转 Date
     *
     * @param localDate localDate
     * @param localTime localTime
     * @return Date
     */
    public static Date toDate(LocalDate localDate, LocalTime localTime) {
        LocalDateTime localDateTime = null;

        if (localTime == null) {
            localDateTime = localDate.atStartOfDay();
        } else {
            localDateTime = localDate.atTime(localTime);
        }

        return toDate(localDateTime);
    }

    /**
     * 转 Date
     *
     * @param localDateTime localDateTime
     * @return Date
     */
    public static Date toDate(LocalDateTime localDateTime) {
        return Date.from(localDateTime.atZone(getZoneIdDefault()).toInstant());
    }


    /**
     * 转 LocalDate
     *
     * @param date date
     * @return LocalDate
     */
    public static LocalDate toLocalDate(Date date) {
        LocalDateTime localDateTime = toLocalDateTime(date);
        return localDateTime.toLocalDate();
    }

    /**
     * 转 LocalDate
     *
     * @param date date
     * @return LocalDate
     */
    public static LocalTime toLocalTime(Date date) {
        LocalDateTime localDateTime = toLocalDateTime(date);
        return localDateTime.toLocalTime();
    }

    /**
     * 转 LocalDateTime
     *
     * @param date date
     * @return LocalDateTime
     */
    public static LocalDateTime toLocalDateTime(Date date) {
        return LocalDateTime.ofInstant(date.toInstant(), getZoneIdDefault());
    }

    /**
     * 毫秒时间戳
     */
    public static Long toMillisecond(Date date) {
        return date.getTime();
    }

    /**
     * 毫秒时间戳
     */
    public static Long toMillisecond(LocalDateTime localDateTime) {
        return toMillisecond(localDateTime, getZoneIdDefault());
    }

    /**
     * 毫秒时间戳
     */
    public static Long toMillisecond(LocalDateTime localDateTime, ZoneId zoneId) {
        return localDateTime.atZone(zoneId).toInstant().toEpochMilli();
    }

    /**
     * 几天前
     *
     * @param days 天数
     * @return LocalDateTime
     */
    public static LocalDateTime minusDays(long days) {
        return nowLocalDateTime().minusDays(days);
    }

    /**
     * 几天前
     *
     * @param localDateTime 日期和时间
     * @param days          天数
     * @return LocalDateTime
     */
    public static LocalDateTime minusDays(LocalDateTime localDateTime, long days) {
        return localDateTime.minusDays(days);
    }

    /**
     * 几天后
     *
     * @param days 天数
     * @return LocalDateTime
     */
    public static LocalDateTime plusDays(long days) {
        return nowLocalDateTime().plusDays(days);
    }

    /**
     * 几天后
     *
     * @param localDateTime 日期和时间
     * @param days          天数
     * @return LocalDateTime
     */
    public static LocalDateTime plusDays(LocalDateTime localDateTime, long days) {
        return localDateTime.plusDays(days);
    }

    /**
     * 日期是否一致
     *
     * @param localDate1 日期1
     * @param localDate2 日期2
     * @return 是否一致
     */
    public static boolean isEqual(LocalDate localDate1, LocalDate localDate2) {
        return localDate1.isEqual(localDate2);
    }

    /**
     * 日期和时间是否一致
     *
     * @param localDateTime1 日期时间1
     * @param localDateTime2 日期时间2
     * @return 是否一致
     */
    public static boolean isEqual(LocalDateTime localDateTime1, LocalDateTime localDateTime2) {
        return localDateTime1.isEqual(localDateTime2);
    }

    /**
     * 当前时间，是否在该时间范围内 <br/>
     * 不包括起始时间和截止时间
     *
     * @param beginYear       起始年
     * @param beginMonth      起始月
     * @param beginDayOfMonth 起始日
     * @param endYear         截止年
     * @param endMonth        截止月
     * @param endDayOfMonth   截止日
     * @return 是否
     */
    public static boolean inRangeNowDay(
            int beginYear, int beginMonth, int beginDayOfMonth,
            int endYear, int endMonth, int endDayOfMonth
    ) {
        LocalDate now = nowLocalDate();
        LocalDate begin = LocalDate.of(beginYear, beginMonth, beginDayOfMonth);
        LocalDate end = LocalDate.of(endYear, endMonth, endDayOfMonth);

        return now.isAfter(begin) && now.isBefore(end);
    }

    /**
     * 当前时间，是否在该时间范围内 <br/>
     * 不包括起始时间和截止时间
     *
     * @param beginHour   起始时
     * @param beginMinute 起始分
     * @param beginSecond 起始秒
     * @param endHour     截止时
     * @param endMinute   截止分
     * @param endSecond   截止秒
     * @return 是否
     */
    public static boolean inRangeNowTime(
            int beginHour, int beginMinute, int beginSecond,
            int endHour, int endMinute, int endSecond
    ) {
        LocalTime now = nowLocalTime();
        LocalTime begin = LocalTime.of(beginHour, beginMinute, beginSecond);
        LocalTime end = LocalTime.of(endHour, endMinute, endSecond);

        return now.isAfter(begin) && now.isBefore(end);
    }


    /**
     * 当前时间，是否在该时间范围内 <br/>
     * 不包括起始时间和截止时间
     *
     * @param beginYear       起始年
     * @param beginMonth      起始月
     * @param beginDayOfMonth 起始日
     * @param beginHour       起始时
     * @param beginMinute     起始分
     * @param beginSecond     起始秒
     * @param endYear         截止年
     * @param endMonth        截止月
     * @param endDayOfMonth   截止日
     * @param endHour         截止时
     * @param endMinute       截止分
     * @param endSecond       截止秒
     * @return 是否
     */
    public static boolean inRangeNow(
            int beginYear, int beginMonth, int beginDayOfMonth, int beginHour, int beginMinute, int beginSecond,
            int endYear, int endMonth, int endDayOfMonth, int endHour, int endMinute, int endSecond
    ) {
        LocalDateTime now = nowLocalDateTime();
        Long nowLong = toMillisecond(now);

        LocalDateTime begin = LocalDateTime.of(beginYear, beginMonth, beginDayOfMonth, beginHour, beginMinute, beginSecond);
        Long beginLong = toMillisecond(begin);

        LocalDateTime end = LocalDateTime.of(endYear, endMonth, endDayOfMonth, endHour, endMinute, endSecond);
        Long endLong = toMillisecond(end);

        return beginLong < nowLong && nowLong < endLong;
    }

    /**
     * 相差多少年, 够完整年算1年
     *
     * @param begin 开始
     * @param end   截止
     * @return 结果
     */
    public static long betweenYears(LocalDate begin, LocalDate end) {
        Period period = Period.between(begin, end);
        return period.getYears();
    }

    /**
     * 相差多少月, 够完整月算1月
     *
     * @param begin 开始
     * @param end   截止
     * @return 结果
     */
    public static long betweenMonths(LocalDate begin, LocalDate end) {
        Period period = Period.between(begin, end);
        return period.toTotalMonths();
    }

    /**
     * 相差多少天, 够完整天算1天
     *
     * @param begin 开始
     * @param end   截止
     * @return 结果
     */
    public static long betweenDays(LocalDateTime begin, LocalDateTime end) {
        Duration duration = Duration.between(begin, end);
        return duration.toDays();
    }

    /**
     * 相差多少小时, 够完整小时算1小时
     *
     * @param begin 开始
     * @param end   截止
     * @return 结果
     */
    public static long betweenHours(LocalDateTime begin, LocalDateTime end) {
        Duration duration = Duration.between(begin, end);
        return duration.toHours();
    }

    /**
     * 相差多少分钟
     *
     * @param begin 开始
     * @param end   截止
     * @return 结果
     */
    public static long betweenMinutes(LocalDateTime begin, LocalDateTime end) {
        Duration duration = Duration.between(begin, end);
        return duration.toMinutes();
    }

    /**
     * 获取默认时区
     */
    public static ZoneId getZoneIdDefault() {
        return getZoneIdDefault_02();
    }

    private static ZoneId getZoneIdDefault_01() {
        return ZoneId.of("Asia/Shanghai");
    }

    private static ZoneId getZoneIdDefault_02() {
        return ZoneId.of(ZoneId.SHORT_IDS.get("CTT"));
    }

    /**
     * 打印 ZoneIds
     */
    public static void printZoneIds() {
        Set<String> zoneIds = ZoneId.getAvailableZoneIds(); // Asia/Shanghai
        zoneIds.stream().iterator().forEachRemaining(s -> {
            System.out.println(s);
        });
    }
}
